<?php

namespace Drupal\akismet\Storage;

/**
 * Responsible for managing and retrieving a site's blacklist.
 *
 * It uses the DrupalClient for the Akismet API to handle operations.
 *
 * @todo Remove this, this is a Mollom remnant.
 *   See https://www.drupal.org/project/akismet/issues/3218535.
 */
class BlacklistStorage {

  const TYPE_SPAM = 'spam';
  const TYPE_PROFANITY = 'profanity';
  const TYPE_UNWANTED = "unwanted";

  const CONTEXT_ALL_FIELDS = 'allFields';
  const CONTEXT_AUTHOR_FIELDS = 'author';
  const CONTEXT_AUTHOR_NAME = 'authorName';
  const CONTEXT_AUTHOR_MAIL = 'authorMail';
  const CONTEXT_AUTHOR_IP = 'authorIp';
  const CONTEXT_AUTHOR_ID = 'authorId';
  const CONTEXT_LINKS = 'links';
  const CONTEXT_POST_FIELDS = 'post';
  const CONTEXT_POST_TITLE = 'postTitle';
  const CONTEXT_POST_BODY = 'post';

  const MATCH_EXACT = 'exact';
  const MATCH_CONTAINS = 'contains';

  /**
   * Static blacklist storage.
   *
   * @var array
   */
  static private $blacklist = NULL;

  /**
   * Gets a blacklist of entries limited by type.
   *
   * @param string $type
   *   The type of list to load; TYPE_SPAM, TYPE_PROFANITY, TYPE_UNWANTED.
   *
   * @return array
   *   The list of the specified type.
   */
  public static function getList($type = NULL): array {
    if (!in_array(
      $type,
      [self::TYPE_SPAM, self::TYPE_PROFANITY, self::TYPE_UNWANTED]
    )) {
      $type = NULL;
    }
    $list = self::loadList();
    if (!empty($type)) {
      $list = array_filter($list, function ($entry) use ($type) {
        return $entry['reason'] === $type;
      });
    }
    return $list;
  }

  /**
   * Gets a blacklist entry.
   *
   * @param string $entry_id
   *   The id of the blacklist entry to retrieve.
   *
   * @return array
   *   Associative array of blacklist entry data.
   */
  public static function getEntry(string $entry_id): array {
    return \Drupal::service('akismet.client')->getBlacklistEntry($entry_id);
  }

  /**
   * Saves a blacklist entry on a blacklist.
   *
   * Note: currently only adding is supported by the API, not updating.
   *
   * @param array $entry
   *   An associative array of blacklist entry data.
   *   - created: A timestamp in seconds since the UNIX epoch of when the entry
   *     was created.
   *   - value: The blacklisted string/value.
   *   - reason: A string denoting the reason for why the value is blacklisted;
   *     one of 'spam', 'profanity', 'quality', or 'unwanted'. Defaults to
   *     'unwanted'.
   *   - context: A string denoting where the entry's value may match; one of
   *     'allFields', 'links', 'authorName', 'authorMail', 'authorIp',
   *     'authorIp', or 'postTitle'. Defaults to 'allFields'.
   *   - match: A string denoting how precise the entry's value may match; one
   *     of 'exact' or 'contains'. Defaults to 'contains'.
   *   - status: An integer denoting whether the entry is enabled (1) or not
   *     (0).
   *   - note: A custom string explaining the entry. Useful in a multi-moderator
   *     scenario.
   *
   * @return bool
   *   True if successful, false if not.
   */
  public static function saveEntry(array $entry): bool {
    $result = \Drupal::service('akismet.client')->saveBlacklistEntry($entry);
    return !empty($result['id']);
  }

  /**
   * Deletes a blacklist entry.
   *
   * @param string $id
   *   The id of the blacklist entry to delete.
   *
   * @response bool
   *   True if successful, false if not.
   */
  public static function deleteEntry(string $id) {
    \Drupal::service('akismet.client')->deleteBlacklistEntry($id);
  }

  /**
   * Loads the blacklist for this site (all types).
   *
   * @return array
   *   The array of blacklist entries for this site.
   *
   * @todo Add error handling.
   */
  public static function loadList(): array {
    if (is_null(self::$blacklist)) {
      self::$blacklist = \Drupal::service('akismet.client')->getBlacklist();
    }
    return self::$blacklist;
  }

}
