<?php

namespace Drupal\akismet\Utility;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Logger\RfcLogLevel;

/**
 * Class Logger.
 *
 * Static utility functions regarding logging.
 *
 * @package Drupal\akismet\Utility
 */
class Logger {

  /**
   * Adds a log entry to a global log (per-request).
   *
   * The Akismet client may perform multiple requests, and the client is able to
   * recover from certain errors. The details of each request are important for
   * support and debugging, but individual log messages for each request are too
   * much and would confuse users, especially when (false-)errors appear in
   * between.
   *
   * Therefore, the Akismet module collects all messages generated by the module
   * integration code as well as by the Akismet client class within a single
   * request, and only logs a single message when the request ends.
   *
   * This collection expects that at least one entry is logged that contains the
   * primary log message and its severity.
   *
   * @param array $entry
   *   (optional) An associative array describing the entry to add to the log.
   *   If supplied, the special keys 'message' and 'arguments' are taken over as
   *   primary log message. All other key/value pairs will be appended to the
   *   resulting log message, whereas the key denotes a label/heading and the
   *   value is var_export()ed afterwards, unless NULL.
   * @param int|null $severity
   *   (optional) The severity of the primary log message, as per RFC 3164.
   *   Possible values are WATCHDOG_ERROR, WATCHDOG_WARNING, etc. See watchdog()
   *   for details. Defaults to WATCHDOG_NOTICE when a 'message' is passed.
   * @param bool $reset
   *   (optional) Whether to empty the log and return its contents.
   *
   * @return array
   *   An associative array containing the log:
   *   - message: The primary log message.
   *   - arguments: An array of placeholder token replacement values for
   *     _akismet_format_string().
   *   - severity: The severity of the primary log message.
   *   - entries: A list of all $entry items that have been passed in.
   *
   * @see akismet_exit()
   */
  public static function addMessage(array $entry = NULL, int $severity = NULL, bool $reset = FALSE): array {
    // Start with debug severity level.
    $log = &drupal_static(__FUNCTION__, []);

    if ($reset) {
      $return = $log;
      $log = [];
      return $return;
    }
    if (!isset($entry)) {
      return $log;
    }
    // Take over the primary message.
    // Only the module integration code sets a message.
    if (isset($entry['message'])) {
      $log['message'] = $entry['message'];
      $log['arguments'] = $entry['arguments'] ?? [];

      // Default to notice severity for module messages.
      if (!isset($severity)) {
        $severity = RfcLogLevel::NOTICE;
      }
    }

    if (!isset($log['severity'])) {
      $log['severity'] = RfcLogLevel::DEBUG;
    }
    // Update severity, if the entry is more severe than existing.
    // Fail-over handling for requests is encapsulated in the Akismet class,
    // which only passes the final severity already.
    if (isset($severity) && $severity < $log['severity']) {
      $log['severity'] = $severity;
    }

    $log['entries'][] = $entry;

    return $log;
  }

  /**
   * Logs a single system message.
   *
   * Potentially contains multiple Akismet log entries.
   *
   * @see akismet_log()
   * @see _akismet_format_log()
   * @see watchdog()
   */
  public static function writeLog() {
    $config = \Drupal::config('akismet.settings');
    // Retrieve the log and reset it.
    $log = static::addMessage(NULL, NULL, TRUE);
    if (empty($log)) {
      return;
    }

    // Only log if severity meets configured minimum severity, or if testing
    // mode is enabled.
    if ($config->get('test_mode.enabled') || $log['severity'] < $config->get('log_level')) {
      list($message, $arguments) = static::formatLog($log);
      \Drupal::logger('akismet')->log($log['severity'], $message, $arguments);
    }
  }

  /**
   * Log a Akismet system message.
   *
   * In essence, this is a poor man's YAML implementation (give or take some
   * details, but note especially the indentation for array sub-elements).
   *
   * @param array $log
   *   A list of message parts. Each item is an associative array whose keys are
   *   log message strings and whose corresponding values are t()-style
   *   replacement token arguments. At least one part is required.
   *
   * @return array
   *   Formatted log; array contains the message and an array of t()-style
   *   replacement tokens.
   */
  protected static function formatLog(array $log): array {
    $message = $log['message'] ?? '';
    $arguments = $log['arguments'] ?? [];

    // Hide further message details in the log overview table, if any.
    // @see theme_dblog_message()
    if (!empty($log['entries'])) {
      // A <br> would be more appropriate, but filter_xss_admin() does not allow
      // it.
      $message = $message . "<p>\n\n</p>";
    }

    // Walk through each log entry to prepare and format its message and
    // arguments.
    $i = 0;
    foreach ($log['entries'] as $entry) {
      // Take over message and arguments literally (if any).
      if (isset($entry['message'])) {
        $message .= '<p>';
        if (!empty($entry['arguments'])) {
          $message .= new FormattableMarkup($entry['message'], $entry['arguments']);
          unset($entry['arguments']);
        }
        else {
          $message .= $entry['message'];
        }
        unset($entry['message']);
        $message .= "</p>\n";
      }
      unset($entry['severity']);

      // Prettify replacement token values, if possible.
      foreach ($entry as $token => $array) {
        // Only prettify non-scalar values plus Booleans.
        // I.e., NULL, TRUE, FALSE, array, and object.
        if (is_scalar($array) && !is_bool($array)) {
          $value = $array;
        }
        else {
          $flat_value = NULL;
          // Convert arrays and objects.
          // @todo Objects?
          if (isset($array) && !is_scalar($array)) {
            $ref = &$array;
            $key = key($ref);
            $parents = [];
            $flat_value = '';
            while ($key !== NULL) {
              if (is_scalar($ref[$key]) || is_bool($ref[$key]) || is_null($ref[$key])) {
                $value = var_export($ref[$key], TRUE);
                // Indent all values to have a visual separation from the last.
                $flat_value .= str_repeat('  ', count($parents) + 1) . "$key = $value\n";
              }
              if (is_array($ref[$key])) {
                $flat_value .= str_repeat('  ', count($parents) + 1) . "$key =\n";
              }

              // Recurse into nested keys, if the current key is not scalar.
              if (is_array($ref[$key]) && !empty($ref[$key])) {
                $parents[] = &$ref;
                $ref = &$ref[$key];
                $key = key($ref);
              }
              else {
                // Move to next key if there is one.
                next($ref);
                if (key($ref) !== NULL) {
                  $key = key($ref);
                }
                // Move back to parent key, if there is one.
                elseif ($parent = array_pop($parents)) {
                  $ref = &$parent;
                  next($ref);
                  $key = key($ref);
                }
                // Otherwise, reached the end of array and recursion.
                else {
                  $key = NULL;
                }
              }
            }
          }
          $value = NULL;
          // Use prettified string representation.
          if ($flat_value !== NULL) {
            $value = $flat_value;
          }
          // Use var_export() for Booleans.
          // Do not output NULL values on the top-level to allow for labels
          // without following value.
          elseif ($array !== NULL) {
            $value = var_export($array, TRUE);
          }
        }

        // Inject all other key/value pairs as @headingN (and optional
        // '<pre>@valueN</pre>') placeholders.
        if (isset($value)) {
          $message .= "@heading{$i}\n<pre>@value{$i}</pre>\n";
          $arguments += [
            '@heading' . $i => $token,
            '@value' . $i => $value,
          ];
        }
        else {
          $message .= "<p>@heading{$i}</p>\n";
          $arguments += [
            '@heading' . $i => $token,
          ];
        }
        $i++;
      }
    }

    return [$message, $arguments];
  }

}
