<?php

namespace Drupal\alert_telegram\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying Telegram messages in the admin interface.
 */
class MessagesController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a MessagesController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection service.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * Displays a list of messages received from Telegram users.
   *
   * @return array
   *   A render array representing the table of messages.
   */
  public function listMessages() {
    $header = [
      'id' => [
        'data' => $this->t('ID'),
        'field' => 'id',
      ],
      'chat_id' => [
        'data' => $this->t('Chat ID'),
        'field' => 'chat_id',
      ],
      'message' => [
        'data' => $this->t('Message'),
        'field' => 'message',
      ],
      'message_type' => [
        'data' => $this->t('Type'),
        'field' => 'message_type',
      ],
      'created' => [
        'data' => $this->t('Date'),
        'field' => 'created',
        'sort' => 'desc',
      ],
    ];

    $query = $this->database->select('alert_telegram_messages', 'atm')
      ->fields('atm')
      ->extend('Drupal\Core\Database\Query\TableSortExtender')
      ->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(20);

    $results = $query->orderByHeader($header)->execute();
    $rows = [];

    foreach ($results as $row) {
      $rows[] = [
        'id' => $row->id,
        'chat_id' => $row->chat_id,
        'message' => $row->message,
        'message_type' => ucfirst($row->message_type),
        'created' => \Drupal::service('date.formatter')->format($row->created, 'short'),
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No messages found.'),
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

}
