<?php

namespace Drupal\alert_telegram\Service;

use GuzzleHttp\ClientInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for interacting with the Telegram Bot API.
 */
class TelegramClient {

  /**
   * The HTTP client for making requests to the Telegram API.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Telegram bot token.
   *
   * @var string
   */
  protected $botToken;

  /**
   * The Telegram API URL.
   *
   * @var string
   */
  protected $apiUrl;

  /**
   * The logger channel for alert_telegram.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a TelegramClient object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client for making requests.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   *
   * @throws \Exception
   *   Thrown when the bot token is not configured.
   */
  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('alert_telegram');

    $config = $this->configFactory->get('alert_telegram.settings');
    $this->botToken = $config->get('telegram_bot_token');

    if (empty($this->botToken)) {
      $this->logger->error('Telegram bot token is not configured.');
      throw new \Exception('Telegram bot token is not configured.');
    }

    $this->apiUrl = 'https://api.telegram.org/bot' . $this->botToken . '/';
  }

  /**
   * Sends a message via Telegram.
   *
   * @param int $chat_id
   *   The chat ID to send the message to.
   * @param string $message
   *   The message text to send.
   *
   * @return array|null
   *   The response data from Telegram API, or NULL if an error occurred.
   */
  public function sendMessage($chat_id, $message) {
    try {
      $response = $this->httpClient->post($this->apiUrl . 'sendMessage', [
        'json' => [
          'chat_id' => $chat_id,
          'text' => $message,
        ],
      ]);

      $data = json_decode($response->getBody(), TRUE);

      if (!$data['ok']) {
        $this->logger->error('Failed to send message to chat ID @chat_id: @description', [
          '@chat_id' => $chat_id,
          '@description' => $data['description'],
        ]);
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Exception when sending message to chat ID @chat_id: @message', [
        '@chat_id' => $chat_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
