<?php

namespace Drupal\alert_telegram\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Component\Serialization\Json;
use Drupal\alert_telegram\Service\TelegramClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Database\Connection;

/**
 * Controller for handling Telegram webhook requests.
 */
class WebhookController extends ControllerBase {

  /**
   * The Telegram client service.
   *
   * @var \Drupal\alert_telegram\Service\TelegramClient
   */
  protected $telegramClient;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel for this module.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The mail manager service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * The database connection service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a WebhookController object.
   *
   * @param \Drupal\alert_telegram\Service\TelegramClient $telegram_client
   *   The Telegram client service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory service.
   * @param \Drupal\Core\Mail\MailManagerInterface $mail_manager
   *   The mail manager service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection service.
   */
  public function __construct(TelegramClient $telegram_client, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, MailManagerInterface $mail_manager, Connection $database) {
    $this->telegramClient = $telegram_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('alert_telegram');
    $this->mailManager = $mail_manager;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('alert_telegram.telegram_client'),
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('plugin.manager.mail'),
      $container->get('database')
    );
  }

  /**
   * Handles incoming webhook requests from Telegram.
   *
   * @param string $secret
   *   The secret key for validating the webhook.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming HTTP request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response.
   */
  public function handle($secret, Request $request) {
    $config = $this->configFactory->get('alert_telegram.settings');
    $webhook_secret = $config->get('webhook_secret');

    if ($secret !== $webhook_secret) {
      $this->logger->warning($this->t('Invalid webhook secret provided.'));
      return new Response($this->t('Access Denied'), Response::HTTP_FORBIDDEN);
    }

    $content = $request->getContent();

    if (!$content) {
      return new Response($this->t('No content'), Response::HTTP_BAD_REQUEST);
    }

    $update = Json::decode($content);

    if (!$update) {
      return new Response($this->t('Invalid JSON'), Response::HTTP_BAD_REQUEST);
    }

    if (isset($update['message'])) {
      $message = $update['message'];
      $chat_id = $message['chat']['id'];
      $text = $message['text'] ?? '';
      $message_id = $message['message_id'];

      $text = trim($text);

      switch ($text) {
        case '/start':
          $message_text = <<<EOT
Welcome! Use the following commands:
Ask a question: /ask
Send your feedback: /feedback
Subscribe to important news and information: /subscribe
Unsubscribe: /unsubscribe
EOT;
          $this->telegramClient->sendMessage($chat_id, $this->t($message_text), $message_id);
          break;

        case '/ask':
          $this->telegramClient->sendMessage($chat_id, $this->t('Please, ask your question.'));
          $this->saveUserState($chat_id, 'awaiting_question');
          break;

        case '/feedback':
          $this->telegramClient->sendMessage($chat_id, $this->t('Please, leave your feedback.'));
          $this->saveUserState($chat_id, 'awaiting_feedback');
          break;

        case '/subscribe':
          $this->addSubscriber($chat_id);
          $this->telegramClient->sendMessage($chat_id, $this->t('You have successfully subscribed to notifications.'));
          break;

        case '/unsubscribe':
          $this->removeSubscriber($chat_id);
          $this->telegramClient->sendMessage($chat_id, $this->t('You have unsubscribed from notifications.'));
          break;

        default:
          $state = $this->getUserState($chat_id);
          if ($state === 'awaiting_question') {
            $thank_you_message = <<<EOT
Thank you for your question!
We will get back to you shortly.
EOT;
            $this->saveMessage($chat_id, $text, 'question', $message_id);
            $this->notifyAdmin($chat_id, $text, $this->t('Question'));
            $this->telegramClient->sendMessage($chat_id, $this->t($thank_you_message), $message_id);
            $this->clearUserState($chat_id);
          }
          elseif ($state === 'awaiting_feedback') {
            $this->saveMessage($chat_id, $text, 'feedback', $message_id);
            $this->notifyAdmin($chat_id, $text, $this->t('Feedback'));
            $this->telegramClient->sendMessage($chat_id, $this->t('Thank you for your feedback!'), $message_id);
            $this->clearUserState($chat_id);
          }
          else {
            $unknown_command_message = <<<EOT
Sorry, I do not understand this command.
Please use the menu commands.
EOT;
            $this->telegramClient->sendMessage($chat_id, $this->t($unknown_command_message));
          }
          break;
      }
    }

    return new Response('OK', Response::HTTP_OK);
  }

  /**
   * Adds a subscriber to the database.
   *
   * @param int|string $chat_id
   *   The Telegram chat ID of the subscriber.
   */
  private function addSubscriber($chat_id) {
    $this->database->merge('alert_telegram_subscribers')
      ->key(['chat_id' => $chat_id])
      ->execute();
  }

  /**
   * Removes a subscriber from the database.
   *
   * @param int|string $chat_id
   *   The Telegram chat ID of the subscriber.
   */
  private function removeSubscriber($chat_id) {
    $this->database->delete('alert_telegram_subscribers')
      ->condition('chat_id', $chat_id)
      ->execute();
  }

  /**
   * Saves a message from the user to the database.
   *
   * @param int|string $chat_id
   *   The Telegram chat ID of the user.
   * @param string $text
   *   The message text.
   * @param string $type
   *   The type of message (e.g., 'question', 'feedback').
   * @param int|string $message_id
   *   The Telegram message ID.
   */
  private function saveMessage($chat_id, $text, $type, $message_id) {
    $this->database->insert('alert_telegram_messages')
      ->fields([
        'chat_id' => $chat_id,
        'message' => $text,
        'message_type' => $type,
        'created' => \Drupal::time()->getRequestTime(),
        'message_id' => $message_id,
      ])
      ->execute();
  }

  /**
   * Notifies the administrator via email about a new message.
   *
   * @param int|string $chat_id
   *   The Telegram chat ID of the user.
   * @param string $text
   *   The message text.
   * @param string $type
   *   The type of message (e.g., 'Question', 'Feedback').
   */
  private function notifyAdmin($chat_id, $text, $type) {
    // Retrieve configured notification emails from alert_telegram settings.
    $notification_emails = $this->configFactory->get('alert_telegram.settings')->get('notification_emails');

    if (empty($notification_emails)) {
      // Log an error if notification emails are not configured.
      $this->logger->error($this->t('Notification emails are not configured. Unable to send email notifications.'));
      return;
    }

    // Split the notification emails by newline.
    $emails = array_map('trim', preg_split('/\r\n|\r|\n/', $notification_emails));

    // Get the base URL of the site.
    $base_url = \Drupal::request()->getSchemeAndHttpHost();

    // Build the full URL to the Telegram messages page.
    $telegram_messages_url = $base_url . '/admin/reports/telegram-messages';

    // Prepare email subject.
    $params['subject'] = $this->t('New @type from user', ['@type' => $type]);

    // Prepare the message in HTML format with a clickable link.
    $params['message'] = $this->t('A new message (@type) has been received from user (Chat ID: @chat_id):', [
        '@type' => $type,
        '@chat_id' => $chat_id,
      ]) . "\n" . \Drupal\Component\Utility\Html::escape($text) . "\n" .
      $this->t('Go to Telegram messages page: :url', [
        ':url' => $telegram_messages_url,
      ]);

    // Specify that the email should be sent in HTML format.
    $params['format'] = 'html';

    $langcode = $this->currentUser()->getPreferredLangcode();
    $send = TRUE;

    // Send the email to each configured address.
    foreach ($emails as $to) {
      $this->logger->info($this->t('Attempting to send email to: @to', ['@to' => $to]));

      $result = $this->mailManager->mail('alert_telegram', 'user_message', $to, $langcode, $params, NULL, $send);

      // Log the result of each email attempt.
      if ($result['result'] !== TRUE) {
        $this->logger->error($this->t('Failed to send email to @to.', ['@to' => $to]));
      }
      else {
        $this->logger->info($this->t('Email successfully sent to @to.', ['@to' => $to]));
      }
    }
  }

  /**
   * Saves the user's state (e.g., awaiting input).
   *
   * @param int|string $chat_id
   *   The Telegram chat ID of the user.
   * @param string $state
   *   The state to save (e.g., 'awaiting_question', 'awaiting_feedback').
   */
  private function saveUserState($chat_id, $state) {
    \Drupal::state()->set('alert_telegram_user_state_' . $chat_id, $state);
  }

  /**
   * Retrieves the user's state.
   *
   * @param int|string $chat_id
   *   The Telegram chat ID of the user.
   *
   * @return string|null
   *   The saved state or NULL if not set.
   */
  private function getUserState($chat_id) {
    return \Drupal::state()->get('alert_telegram_user_state_' . $chat_id);
  }

  /**
   * Clears the user's state.
   *
   * @param int|string $chat_id
   *   The Telegram chat ID of the user.
   */
  private function clearUserState($chat_id) {
    \Drupal::state()->delete('alert_telegram_user_state_' . $chat_id);
  }

}
