<?php

namespace Drupal\alert_telegram\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Component\Serialization\Json;
use Drupal\alert_telegram\Service\TelegramClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Database\Connection;

/**
 * Controller for handling Telegram webhook requests.
 */
class WebhookController extends ControllerBase {

  /**
   * The Telegram client service.
   *
   * @var \Drupal\alert_telegram\Service\TelegramClient
   */
  protected $telegramClient;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel for this module.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The mail manager service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * The database connection service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a WebhookController object.
   */
  public function __construct(TelegramClient $telegram_client, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, MailManagerInterface $mail_manager, Connection $database) {
    $this->telegramClient = $telegram_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('alert_telegram');
    $this->mailManager = $mail_manager;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('alert_telegram.telegram_client'),
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('plugin.manager.mail'),
      $container->get('database')
    );
  }

  /**
   * Handles incoming webhook requests from Telegram.
   */
  public function handle($secret, Request $request) {
    $config = $this->configFactory->get('alert_telegram.settings');
    $webhook_secret = $config->get('webhook_secret');

    if ($secret !== $webhook_secret) {
      $this->logger->warning($this->t('Invalid webhook secret provided.'));
      return new Response('Access Denied', Response::HTTP_FORBIDDEN);
    }

    $content = $request->getContent();

    if (!$content) {
      return new Response('No content', Response::HTTP_BAD_REQUEST);
    }

    $update = Json::decode($content);

    if (!$update) {
      return new Response('Invalid JSON', Response::HTTP_BAD_REQUEST);
    }

    if (isset($update['message'])) {
      $message = $update['message'];
      $chat_id = $message['chat']['id'];
      $text = $message['text'] ?? '';

      $text = trim($text);

      switch ($text) {
        case '/start':
          $this->telegramClient->sendMessage($chat_id, $this->t('Welcome! Use the commands /ask, /feedback, /subscribe, /unsubscribe.'));
          break;

        case '/ask':
          $this->telegramClient->sendMessage($chat_id, $this->t('Please ask your question.'));
          $this->saveUserState($chat_id, 'awaiting_question');
          break;

        case '/feedback':
          $this->telegramClient->sendMessage($chat_id, $this->t('Please leave your feedback.'));
          $this->saveUserState($chat_id, 'awaiting_feedback');
          break;

        case '/subscribe':
          $this->addSubscriber($chat_id);
          $this->telegramClient->sendMessage($chat_id, $this->t('You have successfully subscribed to notifications.'));
          break;

        case '/unsubscribe':
          $this->removeSubscriber($chat_id);
          $this->telegramClient->sendMessage($chat_id, $this->t('You have unsubscribed from notifications.'));
          break;

        default:
          $state = $this->getUserState($chat_id);
          if ($state === 'awaiting_question') {
            $this->saveMessage($chat_id, $text, 'question');
            $this->notifyAdmin($chat_id, $text, 'Question');
            $this->telegramClient->sendMessage($chat_id, $this->t('Thank you for your question! We will get back to you soon.'));
            $this->clearUserState($chat_id);
          }
          elseif ($state === 'awaiting_feedback') {
            $this->saveMessage($chat_id, $text, 'feedback');
            $this->notifyAdmin($chat_id, $text, 'Feedback');
            $this->telegramClient->sendMessage($chat_id, $this->t('Thank you for your feedback!'));
            $this->clearUserState($chat_id);
          }
          else {
            $this->telegramClient->sendMessage($chat_id, $this->t('Sorry, I do not understand this command.'));
          }
          break;
      }
    }

    return new Response('OK', Response::HTTP_OK);
  }

  /**
   * Adds a subscriber to the database.
   */
  private function addSubscriber($chat_id) {
    $this->database->merge('alert_telegram_subscribers')
      ->key(['chat_id' => $chat_id])
      ->execute();
  }

  /**
   * Removes a subscriber from the database.
   */
  private function removeSubscriber($chat_id) {
    $this->database->delete('alert_telegram_subscribers')
      ->condition('chat_id', $chat_id)
      ->execute();
  }

  /**
   * Saves a message from the user to the database.
   */
  private function saveMessage($chat_id, $text, $type) {
    $this->database->insert('alert_telegram_messages')
      ->fields([
        'chat_id' => $chat_id,
        'message' => $text,
        'message_type' => $type,
        'created' => time(),
      ])
      ->execute();
  }

  /**
   * Notifies the administrator via email about a new message.
   */
  private function notifyAdmin($chat_id, $text, $type) {
    // Get the site admin email from the site settings.
    $to = $this->configFactory->get('system.site')->get('mail');

    // Log an error if the email is not configured.
    if (empty($to)) {
      $this->logger->error($this->t('The site email is not configured. Unable to send email notifications.'));
      return;
    }

    // Log the admin email.
    $this->logger->info($this->t('Sending email to: @to', ['@to' => $to]));

    // Prepare email parameters.
    $module = 'alert_telegram';
    $key = 'user_message';
    $params['subject'] = $this->t('New @type from user', ['@type' => $type]);
    $params['message'] = $this->t('A new @type has been received from a user (Chat ID: @chat_id): @message', [
      '@type' => $type,
      '@chat_id' => $chat_id,
      '@message' => $text,
    ]);

    $langcode = $this->currentUser()->getPreferredLangcode();
    $send = TRUE;

    // Send the email.
    $result = $this->mailManager->mail($module, $key, $to, $langcode, $params, NULL, $send);

    if ($result['result'] !== TRUE) {
      $this->logger->error($this->t('Failed to send email to @to.', ['@to' => $to]));
    }
    else {
      $this->logger->info($this->t('Email successfully sent to @to.', ['@to' => $to]));
    }
  }

  /**
   * Saves the user's state (e.g., awaiting input).
   */
  private function saveUserState($chat_id, $state) {
    \Drupal::state()->set('alert_telegram_user_state_' . $chat_id, $state);
  }

  /**
   * Retrieves the user's state.
   */
  private function getUserState($chat_id) {
    return \Drupal::state()->get('alert_telegram_user_state_' . $chat_id);
  }

  /**
   * Clears the user's state.
   */
  private function clearUserState($chat_id) {
    \Drupal::state()->delete('alert_telegram_user_state_' . $chat_id);
  }

}
