<?php

namespace Drupal\alert_telegram\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\alert_telegram\Service\TelegramClient;
use Drupal\Core\Database\Connection;

/**
 * Form for replying to Telegram messages.
 */
class ReplyForm extends FormBase {
  protected $telegramClient;
  protected $database;

  /**
   * Constructs a ReplyForm object.
   */
  public function __construct(TelegramClient $telegram_client, Connection $database) {
    $this->telegramClient = $telegram_client;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('alert_telegram.telegram_client'),
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'alert_telegram_reply_form';
  }

  /**
   * Builds the reply form.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $message = $this->database->select('alert_telegram_messages', 'atm')
      ->fields('atm', ['chat_id', 'message'])
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    $form['chat_id'] = [
      '#type' => 'hidden',
      '#value' => $message->chat_id,
    ];

    $form['reply_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reply message'),
      '#default_value' => '',
      '#required' => TRUE,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send Reply'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $chat_id = $form_state->getValue('chat_id');
    $reply_message = $form_state->getValue('reply_message');
    $reply_to_message_id = $form_state->getValue('message_id'); // ID исходного сообщения

    $this->telegramClient->sendMessage($chat_id, $reply_message, $reply_to_message_id);

    \Drupal::messenger()->addMessage($this->t('Reply sent successfully.'));
    $form_state->setRedirect('alert_telegram.messages_list');
  }

}
