<?php

namespace Drupal\alert_telegram\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\alert_telegram\Service\TelegramClient;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Controller for handling message actions like deleting or replying to messages.
 */
class MessageActionsController extends ControllerBase {
  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The Telegram client service.
   *
   * @var \Drupal\alert_telegram\Service\TelegramClient
   */
  protected $telegramClient;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a MessageActionsController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\alert_telegram\Service\TelegramClient $telegram_client
   *   The Telegram client service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(Connection $database, TelegramClient $telegram_client, MessengerInterface $messenger) {
    $this->database = $database;
    $this->telegramClient = $telegram_client;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('alert_telegram.telegram_client'),
      $container->get('messenger')
    );
  }

  /**
   * Deletes a message from the database.
   *
   * @param int $id
   *   The message ID to delete.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A response redirecting the user after deletion.
   */
  public function deleteMessage($id) {
    if (!$this->currentUser()->hasPermission('administer site configuration')) {
      $this->messenger->addError($this->t('You do not have permission to delete this message. Please contact an administrator.'));
      return new RedirectResponse('/admin/reports/telegram-messages');
    }

    // Check if the message exists before attempting to delete it.
    $message_exists = $this->database->select('alert_telegram_messages', 'atm')
      ->fields('atm', ['id'])
      ->condition('id', $id)
      ->execute()
      ->fetchField();

    if ($message_exists) {
      // Delete the message.
      $this->database->delete('alert_telegram_messages')
        ->condition('id', $id)
        ->execute();
      $this->messenger->addMessage($this->t('Message deleted successfully.'));
    }
    else {
      $this->messenger->addError($this->t('The message could not be found.'));
    }

    return new RedirectResponse('/admin/reports/telegram-messages');
  }

  /**
   * Sends a reply to a user via Telegram.
   *
   * @param int $id
   *   The message ID to reply to.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A response redirecting the user after the reply.
   */
  public function sendReply($id, Request $request) {
    // Check if the message exists and get the chat_id.
    $message = $this->database->select('alert_telegram_messages', 'atm')
      ->fields('atm', ['chat_id', 'message_id'])
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    if ($message && $request->getMethod() === 'POST') {
      $reply_message = $request->request->get('reply_message');
      \Drupal::logger('alert_telegram')->info('Replying to message with ID: @id and message_id: @message_id', [
        '@id' => $id,
        '@message_id' => $message->message_id,
      ]);

      // Send the reply via Telegram.
      $this->telegramClient->sendMessage($message->chat_id, $reply_message, $message->message_id);

      $this->messenger->addMessage($this->t('Reply sent successfully.'));
    }
    else {
      $this->messenger->addError($this->t('Unable to send reply. Please try again.'));
    }

    return new RedirectResponse('/admin/reports/telegram-messages');
  }
}
