<?php

namespace Drupal\alert_telegram\Service;

use GuzzleHttp\ClientInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for interacting with the Telegram Bot API.
 */
class TelegramClient {

  /**
   * The HTTP client for making API requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Telegram bot token.
   *
   * @var string
   */
  protected $botToken;

  /**
   * The base URL for the Telegram Bot API.
   *
   * @var string
   */
  protected $apiUrl;

  /**
   * The logger channel for this module.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a TelegramClient object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client for making requests.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   *
   * @throws \Exception
   *   Throws exception if the Telegram bot token is not configured.
   */
  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('alert_telegram');

    $config = $this->configFactory->get('alert_telegram.settings');
    $this->botToken = $config->get('telegram_bot_token');

    if (empty($this->botToken)) {
      $this->logger->error('Telegram bot token is not configured.');
      throw new \Exception('Telegram bot token is not configured.');
    }

    $this->apiUrl = 'https://api.telegram.org/bot' . $this->botToken . '/';
  }

  /**
   * Sends a message via Telegram.
   *
   * @param int|string $chat_id
   *   The Telegram chat ID to send the message to.
   * @param string $message
   *   The message content to be sent.
   * @param int|null $reply_to_message_id
   *   (optional) The ID of the message to reply to, if any.
   *
   * @return array|null
   *   The response data if the message was sent successfully, NULL otherwise.
   */
  public function sendMessage($chat_id, $message, $reply_to_message_id = NULL) {
    try {
      $params = [
        'chat_id' => $chat_id,
        'text' => $message,
      ];

      // If a reply-to message ID is provided, add it to the parameters.
      if (!empty($reply_to_message_id)) {
        $params['reply_to_message_id'] = $reply_to_message_id;
      }

      $response = $this->httpClient->post($this->apiUrl . 'sendMessage', [
        'json' => $params,
      ]);

      $data = json_decode($response->getBody(), TRUE);

      if (!$data['ok']) {
        $this->logger->error('Failed to send message to chat ID @chat_id: @description', [
          '@chat_id' => $chat_id,
          '@description' => $data['description'],
        ]);
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Exception when sending message to chat ID @chat_id: @message', [
        '@chat_id' => $chat_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
