<?php

namespace Drupal\alert_telegram\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\Component\Serialization\Json;

/**
 * Controller for displaying Telegram messages in the admin interface.
 */
class MessagesController extends ControllerBase {
  protected $database;
  protected $dateFormatter;

  /**
   * Constructs a MessagesController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(Connection $database, DateFormatterInterface $date_formatter) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('date.formatter')
    );
  }

  /**
   * Displays a list of messages received from Telegram users.
   */
  public function listMessages() {
    $header = [
      'id' => ['data' => $this->t('ID'), 'field' => 'id'],
      'chat_id' => ['data' => $this->t('Chat ID'), 'field' => 'chat_id'],
      'message' => ['data' => $this->t('Message'), 'field' => 'message'],
      'message_type' => ['data' => $this->t('Type'), 'field' => 'message_type'],
      'created' => ['data' => $this->t('Date'), 'field' => 'created', 'sort' => 'desc'],
      'operations' => $this->t('Operations'),
    ];

    try {
      $query = $this->database->select('alert_telegram_messages', 'atm')
        ->fields('atm')
        ->extend('Drupal\Core\Database\Query\TableSortExtender')
        ->extend('Drupal\Core\Database\Query\PagerSelectExtender')
        ->limit(20);

      $results = $query->orderByHeader($header)->execute();
    }
    catch (\Exception $e) {
      \Drupal::logger('alert_telegram')->error('Error fetching messages: @message', ['@message' => $e->getMessage()]);
      return [
        '#markup' => $this->t('An error occurred while fetching messages.'),
      ];
    }

    $rows = [];

    foreach ($results as $row) {
      // Generate the reply button with modal and AJAX attributes.
      $reply_url = Url::fromRoute('alert_telegram.reply_form', ['id' => $row->id], [
        'attributes' => [
          'class' => ['btn', 'btn-outline-success', 'use-ajax'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode(['width' => 500]),
        ],
      ]);
      $reply_button = Link::fromTextAndUrl($this->t('Reply'), $reply_url)->toString();

      // Generate the delete button only if the user has the required permission.
      $delete_button = '';
      if ($this->currentUser()->hasPermission('delete telegram messages')) {
        $delete_url = Url::fromRoute('alert_telegram.delete_message', ['id' => $row->id], [
          'attributes' => ['class' => ['btn', 'btn-outline-danger']],
        ]);
        $delete_button = Link::fromTextAndUrl($this->t('Delete'), $delete_url)->toString();
      }

      // Combine buttons for operations column.
      $operations = $reply_button;
      if ($delete_button) {
        $operations .= ' | ' . $delete_button;
      }

      // Add row to table.
      $rows[] = [
        'id' => $row->id,
        'chat_id' => $row->chat_id,
        'message' => $row->message,
        'message_type' => ucfirst($row->message_type),
        'created' => $this->dateFormatter->format($row->created, 'short'),
        'operations' => [
          'data' => [
            '#markup' => $operations,
          ],
        ],
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No messages found.'),
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    // Attach AJAX library for modal functionality.
    $build['#attached']['library'][] = 'core/drupal.dialog.ajax';

    return $build;
  }

}
