<?php

namespace Drupal\alert_telegram\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\NodeType;
use Drupal\user\RoleInterface;
use Drupal\user\Entity\Role;

/**
 * Configuration form for Alert Telegram settings.
 */
class AlertTelegramSettingsForm extends ConfigFormBase {
  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['alert_telegram.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'alert_telegram_settings_form';
  }

  /**
   * Builds the settings form.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('alert_telegram.settings');

    $form['telegram_bot_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Telegram Bot Token'),
      '#default_value' => $config->get('telegram_bot_token'),
      '#required' => TRUE,
      '#description' => $this->t('Enter your Telegram bot token provided by BotFather.'),
    ];

    $form['webhook_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Webhook Secret'),
      '#default_value' => $config->get('webhook_secret'),
      '#required' => TRUE,
      '#description' => $this->t('Enter a secret token to secure the webhook endpoint. This should be a unique and hard-to-guess string.'),
    ];

    $form['webhook_instructions'] = [
      '#type' => 'markup',
      '#markup' => $this->t('
        <div>
        <hr>
          <strong>Webhook Setup:</strong>
          <ol>
            <li>Generate the URL for your webhook by including the secret key:
              <br><code>https://yourdomain.com/alert-telegram/webhook/your_webhook_secret</code>
            </li>
            <li>Set up the webhook by sending a request to Telegram:
              <br><code>https://api.telegram.org/botYOUR_TELEGRAM_BOT_TOKEN/setWebhook?url=YOUR_WEBHOOK_URL</code>
            </li>
          </ol>
          <p>Replace <code>YOUR_TELEGRAM_BOT_TOKEN</code> with your actual bot token, and <code>YOUR_WEBHOOK_URL</code> with the generated URL.</p>
        <hr>
        </div>
      '),
    ];

    $form['telegram_bot_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Telegram Bot URL'),
      '#default_value' => $config->get('telegram_bot_url'),
      '#description' => $this->t('Enter the URL of your Telegram bot (e.g., https://t.me/YourBotName).'),
      '#required' => TRUE,
    ];

    // Update notification_emails field to use a textarea for multiple lines.
    $form['notification_emails'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Notification Emails'),
      '#default_value' => $config->get('notification_emails'),
      '#description' => $this->t('Enter one or more email addresses, each on a new line, to receive notifications when a new message is received in the chatbot.'),
    ];

    // Fetch all roles to create options for the role selection.
    $roles = Role::loadMultiple();
    $role_options = [];
    foreach ($roles as $role) {
      // Exclude special roles like anonymous and authenticated.
      if ($role->id() !== RoleInterface::ANONYMOUS_ID && $role->id() !== RoleInterface::AUTHENTICATED_ID) {
        $role_options[$role->id()] = $role->label();
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Form submission handler.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $notification_emails = $form_state->getValue('notification_emails');

    // Save the role with access to Telegram messages.
    $this->config('alert_telegram.settings')
      ->set('telegram_bot_token', $form_state->getValue('telegram_bot_token'))
      ->set('webhook_secret', $form_state->getValue('webhook_secret'))
      ->set('telegram_bot_url', $form_state->getValue('telegram_bot_url'))
      ->set('notification_emails', $notification_emails)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
