<?php

namespace Drupal\alert_types\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an Alerts service block.
 *
 * @Block(
 *   id = "alerts_block",
 *   admin_label = @Translation("Alerts"),
 * )
 */
class AlertServiceBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs an AlertServiceBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {

    $entity_ids = $this->entityTypeManager->getStorage('alert')->getQuery()
      ->condition('status', 1)
      ->sort('weight', 'ASC')
      ->execute();

    $entities = $this->entityTypeManager->getStorage('alert')->loadMultiple($entity_ids);

    $alerts = [];
    // Make sure that this block cache gets invalidated every time an alert is
    // added, updated or deleted.
    $cache_tags = ['alert_list'];

    foreach ($entities as $entity) {
      // We're doing access here because it's difficult to get the context
      // for content type and current page when loading content via javascript.
      // We're keeping the javascript though because it helps with browser cache
      // and gives us a better experience.
      if ($entity->access('view', \Drupal::currentUser())) {
        $alerts[] = $entity->id();
        $cache_tags = array_merge($cache_tags, $entity->getCacheTags());
      }
    }

    if (empty($alerts)) {
      return [];
    }

    return [
      '#theme' => 'alert_block',
      '#attached' => [
        'library' => ['alert_types/alerts'],
        'drupalSettings' => [
          'alerts' => $alerts,
        ],
      ],
      '#cache' => [
        'tags' => $cache_tags,
        'contexts' => [
          'url.path',
          'user',
        ],
      ],
    ];
  }

}
