<?php

namespace Drupal\alert_types;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeInterface;

/**
 * Defines a class to build a listing of Alert entities.
 *
 * @ingroup alert_types
 */
class AlertListBuilder extends EntityListBuilder implements FormInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entities being listed.
   *
   * @var \Drupal\Core\Entity\EntityInterface[]
   */
  protected $entities = [];

  /**
   * The term storage handler.
   *
   * @var \Drupal\taxonomy\TermStorageInterface
   */
  protected $storageController;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * Constructs an AlertListForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity manager service.
   * @param \Drupal\Core\Render\RendererInterface|null $renderer
   *   The renderer service.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  public function __construct(EntityTypeInterface $entity_type, EntityTypeManagerInterface $entity_type_manager, RendererInterface $renderer = NULL) {
    parent::__construct($entity_type, $entity_type_manager->getStorage($entity_type->id()));
    $this->entityTypeManager = $entity_type_manager;
    $this->storageController = $entity_type_manager->getStorage('alert');
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'alerts_alerts_list';
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    return $this->formBuilder()->getForm($this);
  }

  /**
   * Load alerts sorted by weight.
   *
   * @return array
   *   Return an array of alert entities.
   */
  public function load() {
    $query = $this->storageController->getQuery();
    $query->accessCheck(FALSE);
    $entity_ids = $query->sort('weight', 'asc')
      ->execute();

    $entities = $this->storageController->loadMultiple($entity_ids);

    return $entities;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    // Fetching all the alerts sorted by weight.
    $this->entities = $this->load();

    // Build draggable table.
    $form['alerts'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#empty' => ['#markup' => $this->t('No Alerts Found.')],
      '#attributes' => [
        'id' => 'alerts',
      ],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'weight',
        ],
      ],
    ];
    if (!empty($this->entities)) {
      foreach ($this->entities as $alert) {
        $form['alerts'][$alert->id()] = $this->buildRow($alert);
      }

      $form['actions']['#type'] = 'actions';
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#button_type' => 'primary',
      ];

    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {

    return [
      'label' => $this->t('Name'),
      'type' => $this->t('Alert Type'),
      'status' => $this->t('Status'),
      'changed' => $this->t('Updated'),
      'weight' => $this->t('Weight'),
      'operations' => $this->t('Operations'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /* @var $entity \Drupal\alert_types\Entity\Alert */
    $row['#attributes']['class'][] = 'draggable';
    $row['#weight'] = $entity->getWeight();
    $row['label'] = ['#markup' => $entity->label()];
    $row['type'] = ['#markup' => $this->getBundleLabel($entity)];
    $row['status'] = ['#markup' => $entity->isActive() ? $this->t('Active') : $this->t('Inactive')];
    $row['changed'] = ['#markup' =>  \Drupal::service('date.formatter')->format($entity->getChangedTime(), 'short')];

    // Add weight column.
    $row['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight for @title', ['@title' => $entity->label()]),
      '#title_display' => 'invisible',
      '#default_value' => $entity->getWeight(),
      '#attributes' => ['class' => ['weight']],
    ];

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Intentionally left blank.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    foreach ($form_state->getValue('alerts') as $id => $value) {
      if (isset($this->entities[$id]) && $this->entities[$id]->getWeight() != $value['weight']) {
        // Save entity only when its weight was changed.
        $this->entities[$id]->setWeight($value['weight']);
        $this->entities[$id]->save();
      }
    }
  }

  /**
   * Returns the form builder.
   *
   * @return \Drupal\Core\Form\FormBuilderInterface
   *   The form builder.
   */
  protected function formBuilder() {
    if (!$this->formBuilder) {
      $this->formBuilder = \Drupal::formBuilder();
    }
    return $this->formBuilder;
  }

  /**
   * Get the label of the bundle.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The alert entity.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The label of the alert bundle.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getBundleLabel(EntityInterface $entity) {
    $bundle_type_id = $entity->getEntityType()->getBundleEntityType();
    return $this->entityTypeManager->getStorage($bundle_type_id)->load($entity->bundle())->label();
  }

}
