<?php

namespace Drupal\Tests\alexanders\Functional;

use Drupal\alexanders\Entity\AlexandersOrder;
use Drupal\alexanders\Entity\AlexandersOrderItem;
use Drupal\alexanders\Entity\AlexandersShipment;
use Drupal\Tests\BrowserTestBase;
use GuzzleHttp\Cookie\CookieJar;

/**
 * Functional tests for Alexanders modules site endpoints.
 *
 * @group alexanders
 */
class AlexanderApiSelfTest extends BrowserTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'alexanders',
    'system',
    'datetime',
    'address',
  ];

  private $client;
  private $url;

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();
    $config = $this->config('alexanders.settings');
    $config->set('real_api_key', 'alex-functionaltests');
    $config->save();
    $this->url = $this->baseUrl . '/alexanders';
    AlexandersOrder::create([
      'order_number' => 1,
      'orderItems' => [
        AlexandersOrderItem::create([
          'sku' => $this->randomString(),
          'quantity' => 1,
          'file' => 'example.com',
          'foil' => 'example.com',
        ]),
      ],
      'shipping' => [
        AlexandersShipment::create([
          'method' => 'Test',
          'address' => [],
        ]),
      ],
    ])->save();
  }

  /**
   * Tests endpoints for a 200 OKAY.
   */
  public function testAlexandersSiteEndpoints() {
    $client = $this->getHttpClient();
    $options = [
      'http_errors' => FALSE,
      'headers' => [
        'X-API-KEY' => 'alex-functionaltests',
        'Content-Type' => 'application/json',
      ],
      'cookies' => $this->getGuzzleCookieJar()
    ];

    // Verify we can't GET resources.
    $response = $client->get($this->url . '/printing/1', $options);
    self::assertEquals(405, $response->getStatusCode());

    // With the correct key in live mode.
    $this->makeRequests(200, $client, $options);

    // With the NO key in live mode.
    $options['headers']['X-API-KEY'] = '';
    $this->makeRequests(403, $client, $options);

    // With the wrong key in live mode.
    $options['headers']['X-API-KEY'] = 'alex-wrong-key';
    $this->makeRequests(403, $client, $options);

    // With fix key in live mode to test caching.
    $config = $this->config('alexanders.settings');
    $config->set('real_api_key', 'alex-wrong-key')->save();
    $this->makeRequests(200, $client, $options);

    // Change to sandbox mode but use the live key.
    $config->set('enable_sandbox', TRUE)
      ->set('sandbox_api_key', 'sandbox-key')
      ->save();
    $this->makeRequests(403, $client, $options);

    // With no key in sandbox mode.
    $options['headers']['X-API-KEY'] = '';
    $this->makeRequests(403, $client, $options);

    // With correct key in sandbox mode.
    $options['headers']['X-API-KEY'] = 'sandbox-key';
    $this->makeRequests(200, $client, $options);
  }

  /**
   * Make requests against the API.
   *
   * @param $expected_code
   * @param $client
   * @param $options
   */
  protected function makeRequests($expected_code, $client, $options) {
    // Run through PUT endpoints.
    $response = $client->put($this->url . '/printing/1', ['body' => json_encode(['dueDate' => '2019-01-16T19:53:37.469Z'])] + $options);
    self::assertEquals($expected_code, $response->getStatusCode());

    $response = $client->put($this->url . '/shipped/1', ['body' => json_encode(['trackingNumber' => '123','dateShipped' => '2019-01-16T19:53:37.469Z', 'cost' => 100, 'carrier' => 'test', 'shipMethod' => 'Test'])] + $options);
    self::assertEquals($expected_code, $response->getStatusCode());

    $response = $client->put($this->url . '/error/1', ['body' => json_encode(['message' => 'test', 'itemKey' => '123'])] + $options);
    self::assertEquals($expected_code, $response->getStatusCode());
  }

  /**
   * Creates a new Guzzle CookieJar with a Xdebug cookie if necessary.
   *
   * @return \GuzzleHttp\Cookie\CookieJar
   *   The Guzzle CookieJar.
   */
  protected function getGuzzleCookieJar() {
    // @todo Add xdebug cookie.
    $cookies = $this->extractCookiesFromRequest(\Drupal::request());
    foreach ($cookies as $cookie_name => $values) {
      $cookies[$cookie_name] = $values[0];
    }
    return CookieJar::fromArray($cookies, $this->baseUrl);
  }

}
