<?php

namespace Drupal\Tests\alias_hierarchy\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\menu_link_content\Entity\MenuLinkContent;

/**
 * Test aliases generated by alias_hierarchy.
 *
 * @group alias_hierarchy
 */
class AliasHierarchyAliasesTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['alias_hierarchy'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->drupalCreateContentType([
      'type' => 'page',
      'name' => 'Basic page',
    ]);
    $this->drupalCreateContentType([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $user = $this->drupalCreateUser([
      'administer pathauto',
      'administer site configuration',
      'administer menu',
      'create page content',
      'create article content',
      'create url aliases',
      'edit any page content',
      'edit any article content',
    ]);
    $this->drupalLogin($user);
  }

  /**
   * Test that we can apply alias_hierarchy to specific content types.
   */
  public function testConfig(): void {
    // By default the pattern applies to all content types (pages + articles).
    foreach (['page', 'article'] as $type) {
      $this->drupalGet("node/add/$type");
      $this->assertSession()->statusCodeEquals(200);

      // Our new fields should show up on node interface.
      $this->assertSession()->fieldExists('alias_hierarchy_custom_alias[0][value]');
      $this->assertSession()->fieldExists('alias_hierarchy_custom_alias[0][prefix]');
      $this->assertSession()->fieldDisabled('alias_hierarchy_custom_alias[0][prefix]');

      // The original field provided by the core path module should not show up.
      $this->assertSession()->fieldNotExists('path[0][alias]');
    }

    // Update selection criteria so our pattern only applies to basic pages.
    // This also  ensures our pattern only includes valid tokens.
    $this->drupalGet('admin/config/search/path/patterns/alias_hierarchy');
    $this->assertSession()->statusCodeEquals(200);
    $edit = [
      'bundles[page]' => TRUE,
      'bundles[article]' => FALSE,
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Pattern Alias Hierarchy saved.');

    // Confirm our fields still show up on basic pages.
    $this->drupalGet('node/add/page');
    $this->assertSession()->fieldExists('alias_hierarchy_custom_alias[0][value]');
    $this->assertSession()->fieldExists('alias_hierarchy_custom_alias[0][prefix]');
    $this->assertSession()->fieldNotExists('path[0][alias]');

    // Confirm our fields do not show up on articles anymore.
    $this->drupalGet('node/add/article');
    $this->assertSession()->fieldNotExists('alias_hierarchy_custom_alias[0][value]');
    $this->assertSession()->fieldNotExists('alias_hierarchy_custom_alias[0][prefix]');
    $this->assertSession()->fieldExists('path[0][alias]');
  }

  /**
   * Test alias generated from custom path, menu link title, or node title.
   */
  public function testAliasSource(): void {
    // Test 1: node with no custom alias, menu link -> alias should be based on
    // the node title.
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    $edit = [
      'title[0][value]' => 'test 1 node title',
      'body[0][value]' => 'test 1 node body',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->statusCodeEquals(200);
    $this->drupalGet('test-1-node-title');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('test 1 node body');

    // Test 2: node with with no custom alias but with a menu link -> alias
    // should be based on the menu link title.
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    $edit = [
      'title[0][value]' => 'Test 2 node title',
      'body[0][value]' => 'Test 2 node body',
      'menu[enabled]' => 1,
      'menu[title]' => 'Test 2 menu link title',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->statusCodeEquals(200);
    $this->drupalGet('test-2-menu-link-title');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Test 2 node body');

    // Test 3: node with a menu link and a custom alias -> alias should be based
    // on the custom alias.
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    $edit = [
      'title[0][value]' => 'Test 3 node title',
      'body[0][value]' => 'Test 3 node body',
      'menu[enabled]' => 1,
      'menu[title]' => 'Test 3 menu link title',
      'alias_hierarchy_custom_alias[0][value]' => 'test 3 custom alias',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->statusCodeEquals(200);
    $this->drupalGet('test-3-custom-alias');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Test 3 node body');
  }

  /**
   * Test custom alias gets cleaned and matches the actual path.
   */
  public function testAliasCleaning(): void {
    // Create custom alias with special characters, upper case letters, accents.
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    $edit = [
      'title[0][value]' => 'Node title',
      'alias_hierarchy_custom_alias[0][value]' => 'Alias/foo\bar#hashé',
    ];
    $this->submitForm($edit, 'Save');

    // Make sure pathauto cleaned up the alias.
    $this->drupalGet('aliasfoobarhashe');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Node title');

    // Make sure our field shows the clean version of the alias.
    $this->drupalGet('node/1/edit');
    $this->assertSession()->fieldValueEquals('alias_hierarchy_custom_alias[0][value]', 'aliasfoobarhashe');
  }

  /**
   * Test alias when the node has parents in the menu hierarchy.
   */
  public function testParents(): void {
    // Create 3 nodes.
    $parent_node = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Parent node title',
    ]);
    $child_node = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Child node title',
      'alias_hierarchy_custom_alias' => 'child custom alias',
    ]);
    $grandchild_node = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Grand-child node title',
    ]);

    // Organize nodes into a menu hierarchy:
    // <main-menu>
    // -- Parent
    // ---- Child
    // ------ Grand-child.
    $parent_menu_link = MenuLinkContent::create([
      'title' => 'Parent Menu link title',
      'link' => ['uri' => 'entity:node/' . $parent_node->id()],
      'menu_name' => 'main',
      'parent' => '',
    ]);
    $parent_menu_link->save();
    $child_menu_link = MenuLinkContent::create([
      'title' => 'Child Menu link title',
      'link' => ['uri' => 'entity:node/' . $child_node->id()],
      'menu_name' => 'main',
      'parent' => $parent_menu_link->getPluginId(),
    ]);
    $child_menu_link->save();
    $grandchild_menu_link = MenuLinkContent::create([
      'title' => 'Grand-child Menu link title',
      'link' => ['uri' => 'entity:node/' . $grandchild_node->id()],
      'menu_name' => 'main',
      'parent' => $child_menu_link->getPluginId(),
    ]);
    $grandchild_menu_link->save();

    // Run cron to update aliases.
    $this->drupalGet('admin/config/system/cron');
    $this->submitForm([], 'Run cron');

    // Test path aliases match the menu hierarchy.
    $this->drupalGet('parent-menu-link-title');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Parent node title');

    $this->drupalGet('parent-menu-link-title/child-custom-alias');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Child node title');

    $this->drupalGet('parent-menu-link-title/child-custom-alias/grand-child-menu-link-title');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Grand-child node title');
  }

}
