<?php

namespace Drupal\alpha_numeric_glossary;

use Drupal\alpha_numeric_glossary\Plugin\views\area\AlphaNumericGlossaryArea;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Utility\Token;
use Drupal\views\Plugin\views\ViewsHandlerInterface;
use Drupal\views\Views;

/**
 * A base views handler for alpha numeric glossary.
 */
class AlphaNumericGlossary {

  use StringTranslationTrait;

  /**
   * The Alpha Numeric Glossary Views Handler.
   *
   * @var \Drupal\views\Plugin\views\ViewsHandlerInterface
   */
  protected $handler;

  /**
   * The Entity Field Manager Interface.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $fieldManager;

  /**
   * The Language ID.
   *
   * @var string
   */
  protected $language;

  /**
   * The ModuleHandler Interface.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The cache backend service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cacheBackend;

  /**
   * The Transliteration Interface.
   *
   * @var \Drupal\Component\Transliteration\TransliterationInterface
   */
  protected $transliteration;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The token replacement instance.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * The url used for links.
   *
   * @var \Drupal\Core\Url
   */
  protected $url;

  /**
   * The translationinterface.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The currentpath.
   *
   * @var \Drupal\Core\Path\CurrentPathStack
   */
  private $currentPath;

  /**
   * Contructs the Alpha Numeric Glossary object.
   *
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $field_manager
   *   The Entity Field Manager Interface.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The ModuleHander Interface.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend service.
   * @param \Drupal\Component\Transliteration\TransliterationInterface $transliteration
   *   A PHPTransliteration object.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   The string translation.
   * @param \Drupal\Core\Path\CurrentPathStack $current_path
   *   The current path.
   */
  public function __construct(EntityFieldManagerInterface $field_manager, LanguageManagerInterface $language_manager, ModuleHandlerInterface $module_handler, CacheBackendInterface $cache_backend, TransliterationInterface $transliteration, Connection $database, Token $token, TranslationInterface $stringTranslation, CurrentPathStack $current_path) {
    $this->fieldManager = $field_manager;
    $this->language = $language_manager->getCurrentLanguage()->getId();
    $this->moduleHandler = $module_handler;
    $this->cacheBackend = $cache_backend;
    $this->transliteration = $transliteration;
    $this->database = $database;
    $this->token = $token;
    $this->setStringTranslation($stringTranslation);
    $this->currentPath = $current_path;
  }

  /**
   * Sets the view handler.
   *
   * @param \Drupal\views\Plugin\views\ViewsHandlerInterface $handler
   *   The views handler.
   */
  public function setHandler(ViewsHandlerInterface $handler) {
    $this->handler = $handler;
  }

  /**
   * {@inheritdoc}
   */
  public function __sleep() {
    $this->_handler = implode(':', [
      $this->handler->view->id(),
      $this->handler->view->current_display,
      $this->handler->areaType,
      $this->handler->realField ?: $this->handler->field,
      $this->language,
    ]);

    return ['_handler'];
  }

  /**
   * {@inheritdoc}
   */
  public function __wakeup() {
    [$name, $display_id, $type, $id, $language] = explode(':', $this->_handler);
    $view = Views::getView($name);
    $view->setDisplay($display_id);
    $this->handler = $view->display_handler->getHandler($type, $id);

    $this->language = $language;
    unset($this->_handler);
  }

  /**
   * Add classes to an attributes array from a view option.
   *
   * @param string[]|string $classes
   *   An array of classes or a string of classes.
   * @param array $attributes
   *   An attributes array to add the classes to, passed by reference.
   *
   * @return array
   *   An array of classes to be used in a render array.
   */
  public function addClasses($classes, array &$attributes) {
    $processed = [];

    // Sanitize any classes provided for the item list.
    foreach ((array) $classes as $v) {
      foreach (array_filter(explode(' ', $v)) as $vv) {
        $processed[] = Html::cleanCssIdentifier($vv);
      }
    }

    // Don't add any classes if it's empty, which will add an empty attribute.
    if ($processed) {
      if (!isset($attributes['class'])) {
        $attributes['class'] = [];
      }
      $attributes['class'] = array_unique(array_merge($attributes['class'], $processed));
    }

    return $classes;
  }

  /**
   * Builds a render array for displaying tokens.
   *
   * @param string $fieldset
   *   The #fieldset name to assign.
   *
   * @return array
   *   The render array for the token info.
   */
  public function buildTokenTree($fieldset = NULL) {
    static $build;

    if (!isset($build)) {
      if ($this->moduleHandler->moduleExists('token')) {
        $build = [
          '#type' => 'container',
          '#title' => $this->t('Browse available tokens'),
        ];
        $build['help'] = [
          '#theme' => 'token_tree_link',
          '#token_types' => ['alpha_numeric_glossary'],
          '#global_types' => TRUE,
          '#dialog' => TRUE,
        ];
      }
      else {
        $build = [
          '#type' => 'fieldset',
          '#title' => $this->t('Available tokens'),
          '#collapsible' => TRUE,
          '#collapsed' => TRUE,
        ];
        $items = [];
        $token_info = alpha_numeric_glossary_token_info();
        foreach ($token_info['tokens'] as $_type => $_tokens) {
          foreach ($_tokens as $_token => $_data) {
            $items[] = "[$_type:$_token] - {$_data['description']}";
          }
        }
        $build['help'] = [
          '#theme' => 'item_list',
          '#items' => $items,
        ];
      }
    }

    return isset($fieldset) ? ['#fieldset' => $fieldset] + $build : $build;
  }

  /**
   * Extract the SQL query from the query information.
   *
   * Once extracted, place it into the options array so it is passed to the
   * render function. This code was lifted nearly verbatim from the views
   * module where the query is constructed for the ui to show the query in the
   * administrative area.
   *
   * @todo Need to make this better somehow?
   */
  public function ensureQuery() {
    if (!$this->getOption('query') && !empty($this->handler->view->build_info['query'])) {
      /** @var \SelectQuery $query */
      $query = $this->handler->view->build_info['query'];
      $quoted = $query->getArguments();
      foreach ($quoted as $key => $val) {
        if (is_array($val)) {
          $quoted[$key] = implode(', ', array_map([
            $this->database,
            'quote',
          ], $val));
        }
        else {
          $quoted[$key] = $this->database->quote($val);
        }
      }
      $this->handler->options['query'] = Html::escape(strtr($query, $quoted));
    }
  }

  /**
   * Retrieves alphabet characters, based on langcode.
   *
   * Note: Do not use range(); always be explicit when defining an alphabet.
   * This is necessary as you cannot rely on the server language to construct
   * proper alphabet characters.
   *
   * @param string $langcode
   *   The langcode to return. If the langcode does not exist, it will default
   *   to English.
   *
   * @return array
   *   An indexed array of alphabet characters, based on langcode.
   *
   * @see hook_alpha_numeric_glossary_alphabet_alter()
   */
  public function getAlphabet($langcode = NULL) {

    // Default (English).
    if ($this->getOption('glossary_case_text') === 'mb_strtolower') {
      $default = [
        'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      ];
    }
    else {
      $default = [
        'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      ];
    }
    static $alphabets;

    // If the langcode is not explicitly specified, default to global langcode.
    if (!isset($langcode)) {
      $langcode = $this->language;
    }

    // Retrieve alphabets.
    if (!isset($alphabets)) {
      // Attempt to retrieve from database cache.
      $cid = "alpha_numeric_glossary:alphabets";
      if (($cache = $this->cacheBackend->get($cid)) && !empty($cache->data)) {
        $alphabets = $cache->data;
      }
      // Build alphabets.
      else {
        // Arabic.
        $alphabets['ar'] = [
          'ا', 'ب', 'ت', 'ث', 'ج', 'ح', 'خ', 'د', 'ذ', 'ر', 'ز', 'س', 'ش', 'ص', 'ض', 'ط', 'ظ', 'ع', 'غ', 'ف', 'ق', 'ك', 'ل', 'م', 'ن', 'و', 'ه', 'ي',
        ];

        // English. Initially the default value, but can be modified in alter.
        $alphabets['en'] = $default;

        // Русский (Russian).
        $alphabets['ru'] = [
          'А', 'Б', 'В', 'Г', 'Д', 'Е', 'Ё', 'Ж', 'З', 'И', 'Й', 'К', 'Л', 'М', 'Н', 'О', 'П', 'Р', 'С', 'Т', 'У', 'Ф', 'Х', 'Ц', 'Ч', 'Ш', 'Щ', 'Ы', 'Э', 'Ю', 'Я',
        ];

        // Allow modules and themes to alter alphabets.
        $this->moduleHandler->alter('alpha_numeric_glossary_alphabet', $alphabets, $this);

        // Cache the alphabets.
        $this->cacheBackend->set($cid, $alphabets);
      }
    }

    // Return alphabet based on langcode.
    return $alphabets[$langcode] ?? $default;
  }

  /**
   * Retrieves all available alpha numeric glossary areas.
   *
   * @param array $types
   *   The handler types to search.
   *
   * @return \Drupal\alpha_numeric_glossary\Plugin\views\area\AlphaNumericGlossaryArea[]
   *   An array of alpha numeric glossary areas.
   */
  public function getAreaHandlers(array $types = ['header', 'footer']) {
    $areas = [];
    foreach ($types as $type) {
      foreach ($this->handler->displayHandler->getHandlers($type) as $handler) {
        if ($handler instanceof AlphaNumericGlossaryArea) {
          $areas[] = $handler;
        }
      }
    }
    return $areas;
  }

  /**
   * Retrieves the characters used to populate the glossary item list.
   *
   * @return \Drupal\alpha_numeric_glossary\AlphaNumericGlossaryCharacter[]
   *   An associative array containing AlphaNumericGlossaryCharacter objects
   *   by its value.
   */
  public function getCharacters() {
    /** @var \Drupal\alpha_numeric_glossary\AlphaNumericGlossaryCharacter[] $characters */
    static $characters;
    $caseText = $this->getOption('glossary_case_text');

    if (!isset($characters)) {
      $all = $this->getOption('glossary_all_display') === '1' ? $this->getOption('glossary_all_label', $this->t('All')) : '';
      $all_value = $this->getOption('glossary_all_value', 'all');
      $numeric_label = $this->getOption('glossary_numeric_label');
      $numeric_type = $this->getOption('glossary_view_numbers', '0');
      $numeric_value = $this->getOption('glossary_numeric_value');
      $numeric_divider = $numeric_type !== '2' && $this->getOption('glossary_numeric_divider') ? ['-' => ''] : [];

      // Add alphabet characters.
      foreach ($this->getAlphabet() as $value) {
        $characters[$value] = $value;
      }

      // Append or prepend numeric item(s).
      $numeric = [];
      if ($numeric_type !== '0') {

        // Determine type of numeric items.
        if ($numeric_type === '2') {
          $numeric[$numeric_value] = Html::escape($numeric_label);
        }
        else {
          foreach ($this->getNumbers() as $value) {
            $numeric[$value] = $value;
          }
        }

        // Merge in numeric items.
        if ($this->getOption('glossary_numeric_position') === 'after') {
          $characters = array_merge($characters, $numeric_divider, $numeric);
        }
        else {
          $characters = array_merge($numeric, $numeric_divider, $characters);
        }
      }

      // Append or prepend the "all" item.
      if ($all) {
        if ($this->getOption('glossary_all_position') === 'before') {
          $characters = [$all_value => $all] + $characters;
        }
        else {
          $characters[$all_value] = $all;
        }
      }

      // Display Count in the objects.
      if ($this->getOption('glossary_display_count')) {
        $getEntityCount = $this->getEntityPrefixes(1, $caseText);
        $charactersMerge = array_merge($characters, $getEntityCount);
        $characters = [];
        foreach ($charactersMerge as $key => $char) {
          $count = 0;
          if (is_int($char)) {
            $count = $char;
          }
          $characters[$key] = $count;
        }
      }

      // Convert characters to objects.
      foreach ($characters as $value => $label) {
        $characters[$value] = new AlphaNumericGlossaryCharacter($this, $label, $value);
      }

      // Determine enabled prefixes.
      $prefixes = $this->getEntityPrefixes(NULL, $caseText);
      foreach ($prefixes as $value) {
        // Ensure numeric prefixes use the numeric label, if necessary.
        if ($this->isNumeric($value) && $numeric_type === '2') {
          $value = $numeric_value;
        }
        if (isset($characters[$value])) {
          $characters[$value]->setEnabled(TRUE);
        }
      }

      // Remove all empty prefixes.
      if (!$this->getOption('glossary_toggle_empty')) {
        // Ensure "all" and numeric divider objects aren't removed.
        if ($all) {
          $prefixes[] = $all_value;
        }
        if ($numeric_divider) {
          $prefixes[] = '-';
        }
        // Determine if numeric results are not empty.
        $characters = array_filter(array_intersect_key($characters, array_flip($prefixes)));
      }

      // Remove all numeric values if they're all empty.
      if ($this->getOption('glossary_numeric_hide_empty')) {
        // Determine if numeric results are not empty.
        $numeric_results = array_filter(array_intersect_key($characters, array_flip($numeric)));
        if (!$numeric_results) {
          $characters = array_diff_key($characters, array_flip($numeric));
        }
      }

      // Default current value to "all", if enabled, or the first character.
      $current = $all ? $all_value : '';

      // Attempt to determine if a valid argument was provided.
      $arg_count = count($this->handler->view->args);
      if ($arg_count) {
        $arg = $caseText($this->handler->view->args[$arg_count - 1]);
        if ($arg && in_array($arg, array_keys($characters))) {
          $current = $arg;
        }
      }

      // Determine the first active character.
      $first_char_active = $this->getOption('glossary_disable_first_char_active');
      if ($first_char_active) {
        $current_path = \Drupal::service('path.current')->getPath();
        if ($current_path == $first_char_active) {
          foreach ($characters as $character) {
            $character->setActive(FALSE);
            break;
          }
        }
        else {
          if ($current) {
            foreach ($characters as $character) {
              if ($character->isNumeric() && $numeric_type === '2' ? $numeric_value === $current : $character->getValue() === $current) {
                $character->setActive(TRUE);
                break;
              }
            }
          }
        }
      }
      else {
        if ($current) {
          foreach ($characters as $character) {
            if ($character->isNumeric() && $numeric_type === '2' ? $numeric_value === $current : $character->getValue() === $current) {
              $character->setActive(TRUE);
              break;
            }
          }
        }
      }
    }

    return $characters;
  }

  /**
   * Construct the actual SQL query for the view being generated.
   *
   * Then parse it to short-circuit certain conditions that may exist and
   * make any alterations. This is not the most elegant of solutions, but it
   * is very effective.
   *
   * @return array
   *   An indexed array of entity identifiers.
   */
  public function getEntityIds() {
    $this->ensureQuery();
    $query_parts = explode("\n", $this->getOption('query'));

    // Get the base field. This will change depending on the type of view we
    // are putting the paginator on and eventually if we are using a
    // Inspired from core/modules/views_ui/src/Form/Ajax/ConfigHandler.php
    // buildForm further exploration needed when chaining relationships.
    // If we are dealing with a substring, then short circuit it as we are most
    // likely dealing with a glossary contextual filter.
    foreach ($query_parts as $k => $part) {
      if ($position = strpos($part, "SUBSTRING")) {
        $part = substr($part, 0, $position) . " 1 OR " . substr($part, $position);
        $query_parts[$k] = $part;
      }
    }

    // Evaluate the last line looking for anything which may limit the result
    // set as we need results against the entire set of data and not just what
    // is configured in the view.
    $last_line = array_pop($query_parts);
    if (substr($last_line, 0, 5) != "LIMIT") {
      $query_parts[] = $last_line;
    }

    // Construct the query from the array and change the single quotes from
    // HTML special characters back into single quotes.
    $query = implode("\n", $query_parts);
    $query = str_replace("&#039;", '\'', $query);
    $query = str_replace("&amp;", '&', $query);
    $query = str_replace("&lt;", '<', $query);
    $query = str_replace("&gt;", '>', $query);
    $query = str_replace("&quot;", '', $query);

    // Based on our query, get the list of entity identifiers that are affected.
    // These will be used to generate the pagination items.
    $entity_ids = [];
    $result = $this->database->query($query);
    while ($data = $result->fetchObject()) {
      $entity_ids[] = $data->nid;
    }
    return $entity_ids;
  }

  /**
   * Retrieve the distinct first character prefix from the field tables.
   *
   * Mark them as TRUE so their pagination item is represented properly.
   *
   * Note that the node title is a special case that we have to take from the
   * node table as opposed to the body or any custom fields.
   *
   * @todo This should be cleaned up more and fixed "properly".
   *
   * @return array
   *   An indexed array containing a unique array of entity prefixes.
   */
  public function getEntityPrefixes($count = NULL, $case_text = 'mb_strtoupper') {
    $prefixes = [];

    if ($entity_ids = $this->getEntityIds()) {
      switch ($this->getOption('glossary_view_field')) {
        case 'name':
          $table = $this->handler->view->storage->get('base_table');
          $where = $this->handler->view->storage->get('base_field');

          // Extract the "name" field from the entity property info.
          $entity_type = $this->handler->view->getBaseEntityType->id();
          $entity_info = $this->fieldManager->getBaseFieldDefinitions($entity_type);
          $field = $entity_info['properties']['name']['schema field'] ?? 'name';
          break;

        case 'title':
          $table = $this->handler->view->storage->get('base_table');
          $where = $this->handler->view->storage->get('base_field');

          // Extract the "title" field from the entity property info.
          $entity_type = $this->handler->view->getBaseEntityType()->id();
          $entity_info = $this->fieldManager->getBaseFieldDefinitions($entity_type);
          $field = $entity_info['properties']['title']['schema field'] ?? 'title';
          break;

        default:
          if (strpos($this->getOption('glossary_view_field'), ':') === FALSE) {
            // Format field name and table for single value fields.
            $field_name = explode('__', $this->getOption('glossary_view_field'));
            $field = $field_name[1] . '_value';
            $table = $this->getOption('glossary_view_field');
          }
          else {
            // Format field name and table for compound value fields.
            [$field_name] = explode('__', $this->getOption('glossary_view_field'), 2);
            $field = str_replace(':', '_', $field_name);
            $field_name_components = explode(':', $field_name);
            $table = $field_name_components[0];
          }
          $where = 'entity_id';
          break;
      }
      if ($count) {
        $result = $this->database->query('SELECT SUBSTR(' . $field . ', 1, 1) AS prefix
                          FROM {' . $table . '}
                          WHERE ' . $where . ' IN ( :nids[] )', [':nids[]' => $entity_ids]);
        while ($data = $result->fetchObject()) {
          $prefixes[] = is_numeric($data->prefix) ? $data->prefix : $case_text($this->transliteration->transliterate($data->prefix));
        }
        $resultArray = array_count_values($prefixes);
      }
      else {
        $result = $this->database->query('SELECT DISTINCT(SUBSTR(' . $field . ', 1, 1)) AS prefix
                          FROM {' . $table . '}
                          WHERE ' . $where . ' IN ( :nids[] )', [':nids[]' => $entity_ids]);
        while ($data = $result->fetchObject()) {
          $prefixes[] = is_numeric($data->prefix) ? $data->prefix : $case_text($this->transliteration->transliterate($data->prefix));
        }
        $resultArray = array_unique(array_filter($prefixes));
      }
    }
    return $resultArray;
  }

  /**
   * Retrieves the proper label for a character.
   *
   * @param mixed $value
   *   The value of the label to retrieve.
   *
   * @return string
   *   The label.
   */
  public function getLabel($value) {
    $characters = $this->getCharacters();

    // Return an appropriate numeric label.
    if ($this->getOption('glossary_view_numbers') === '2' && $this->isNumeric($value)) {
      return $characters[$this->getOption('glossary_numeric_value')]->getLabel();
    }
    elseif (isset($characters[$value])) {
      return $characters[$value]->getLabel();
    }

    // Return the original value.
    return $value;
  }

  /**
   * Retrieves numeric characters, based on langcode.
   *
   * Note: Do not use range(); always be explicit when defining numbers.
   * This is necessary as you cannot rely on the server language to construct
   * proper numeric characters.
   *
   * @param string $langcode
   *   The langcode to return. If the langcode does not exist, it will default
   *   to English.
   *
   * @return array
   *   An indexed array of numeric characters, based on langcode.
   *
   * @see hook_alpha_numberic_glossary_numbers_alter()
   */
  public function getNumbers($langcode = NULL) {
    // Default (English).
    static $default = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    static $numbers;

    // If the langcode is not explicitly specified, default to global langcode.
    if (!isset($langcode)) {
      $langcode = $this->language;
    }

    // Retrieve numbers.
    if (!isset($numbers)) {
      // English. Initially the default value, but can be modified in alter.
      $numbers['en'] = $default;

      // Allow modules and themes to alter numbers.
      $this->moduleHandler->alter('alpha_numeric_glossary_numbers', $numbers, $this);
    }

    // Return numbers based on langcode.
    return $numbers[$langcode] ?? $default;
  }

  /**
   * Retrieves an option from the view handler.
   *
   * @param string $name
   *   The option name to retrieve.
   * @param mixed $default
   *   The default value to return if not set.
   *
   * @return string
   *   The option value or $default if not set.
   */
  public function getOption($name, $default = '') {
    return (string) ($this->handler->options[$name] ?? $default);
  }

  /**
   * Provides the token data that is passed when during replacement.
   *
   * @param string $value
   *   The current character value being processed.
   *
   * @return array
   *   Token data.
   *
   * @see alpha_numeric_glossary_token_info()
   * @see alpha_numeric_glossary_tokens()
   */
  public function getTokens($value = NULL) {
    return [
      'alpha_numeric_glossary' => [
        'path' => $this->getUrl(),
        'value' => $value,
      ],
    ];
  }

  /**
   * Retrieves the URL for the current view.
   *
   * Note: this follows very similarly to \view::get_url to process arguments,
   * however it is in fact severely modified to account for characters appended
   * by this module.
   *
   * @return string
   *   The URL for the view or current_path().
   */
  public function getUrl() {
    if (empty($this->url)) {
      if (!empty($this->handler->view->override_url)) {
        $this->url = $this->handler->view->override_url;
        return $this->url;
      }
      $path = $this->handler->view->getPath();
      $args = $this->handler->view->args;
      // Exclude arguments that were computed, not passed on the URL.
      $position = 0;
      if (!empty($this->handler->view->argument)) {
        foreach ($this->handler->view->argument as $argument) {
          if (!empty($argument->options['default_argument_skip_url'])) {
            unset($args[$position]);
          }
          $position++;
        }
      }

      // Don't bother working if there's nothing to do:
      if (empty($path) || (empty($args) && strpos($path, '%') === FALSE)) {
        $path = $this->currentPath->getPath();
        $pieces = explode('/', $path);
        // If getPath returns heading slash, remove it as it will ba
        // added later.
        if ($pieces[0] == '') {
          array_shift($pieces);
        }
        if (array_key_exists(end($pieces), $this->getCharacters())) {
          array_pop($pieces);
        }
        $this->url = implode('/', $pieces);
        return $this->url;
      }

      $pieces = [];
      $argument_keys = isset($this->handler->view->argument) ? array_keys($this->handler->view->argument) : [];
      $id = current($argument_keys);
      foreach (explode('/', $path) as $piece) {
        if ($piece != '%') {
          $pieces[] = $piece;
        }
        else {
          if (empty($args)) {
            // Try to never put % in a url; use the wildcard instead.
            if ($id && !empty($this->handler->view->argument[$id]->options['exception']['value'])) {
              $pieces[] = $this->handler->view->argument[$id]->options['exception']['value'];
            }
            else {
              // Gotta put something if there just isn't one.
              $pieces[] = '*';
            }
          }
          else {
            $pieces[] = array_shift($args);
          }

          if ($id) {
            $id = next($argument_keys);
          }
        }
      }

      // Just return the computed pieces, don't merge any extra remaining args.
      $this->url = implode('/', $pieces);
    }
    return $this->url;
  }

  /**
   * Retrieves the proper value for a character.
   *
   * @param mixed $value
   *   The value to retrieve.
   *
   * @return string
   *   The value.
   */
  public function getValue($value) {
    $characters = $this->getCharacters();

    // Return an appropriate numeric label.
    if ($this->getOption('glossary_view_numbers') === '2' && $this->isNumeric($value)) {
      return $characters[$this->getOption('glossary_numeric_value')]->getValue();
    }
    elseif (isset($characters[$value])) {
      return $characters[$value]->getLabel();
    }

    // Return the original value.
    return $value;
  }

  /**
   * Determines if value is "numeric".
   *
   * @param string $value
   *   The value to test.
   *
   * @return bool
   *   TRUE or FALSE
   */
  public function isNumeric($value) {
    return ($this->getOption('glossary_view_numbers') === '2' && $value === $this->getOption('glossary_numeric_value')) || in_array($value, $this->getNumbers());
  }

  /**
   * Parses an attribute string saved in the UI.
   *
   * @param string $string
   *   The attribute string to parse.
   * @param array $tokens
   *   An associative array of token data to use.
   *
   * @return array
   *   A parsed attributes array.
   */
  public function parseAttributes($string = NULL, array $tokens = []) {
    $attributes = [];
    if (!empty($string)) {
      $parts = explode(',', $string);
      foreach ($parts as $attribute) {
        if (strpos($attribute, '|') !== FALSE) {
          [$key, $value] = explode('|', $this->token->replace($attribute, $tokens, ['clear' => TRUE]));
          $attributes[$key] = $value;
        }
      }
    }
    return $attributes;
  }

  /**
   * {@inheritdoc}
   */
  public function validate() {
    $name = $this->handler->view->id();
    $display_id = $this->handler->view->current_display;

    // Immediately return if display doesn't have the handler in question.
    $items = $this->handler->view->getHandlers($this->handler->areaType, $display_id);
    $field = $this->handler->realField ?: $this->handler->field;
    if (!isset($items[$field])) {
      return [];
    }

    static $errors = [];
    if (!isset($errors["$name:$display_id"])) {
      $errors["$name:$display_id"] = [];

      // Show an error if not found.
      $areas = $this->getAreaHandlers();
      if (!$areas) {
        $errors["$name:$display_id"][] = $this->t('The view "@name:@display" must have at least one configured alpha pagination area in either the header or footer to use "@field".', [
          '@field' => $field,
          '@name' => $this->handler->view->id(),
          '@display' => $this->handler->view->current_display,
        ]);
      }
      // Show an error if there is more than one instance.
      elseif (count($areas) > 1) {
        $errors["$name:$display_id"][] = $this->t('The view "@name:@display" can only have one configured alpha pagination area in either the header or footer.', [
          '@name' => $this->handler->view->id(),
          '@display' => $this->handler->view->current_display,
        ]);
      }

    }

    return $errors["$name:$display_id"];
  }

}
