<?php

namespace Drupal\alt_text_import_csv\Form;

use Drupal\alt_text_import_csv\AltTextImportBatch;
use Drupal\Component\Utility\Environment;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to import a CSV file to update alt texts.
 */
class AltTextImportForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Creates a AltTextImportForm instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager
  ) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'alt_text_import_csv_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['csv_file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Upload CSV file'),
      '#required' => TRUE,
      '#description' => $this->t('A CSV file whose columns are: page URL, image URL, alt text. The CSV file may contain a header row.'),
      '#upload_location' => 'public://metatag_import_export_csv/',
      '#upload_validators' => [
        'file_validate_extensions' => ['csv'],
        'file_validate_size' => [Environment::getUploadMaxSize()],
      ],
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#required' => TRUE,
      '#value' => $this->t('Import'),
      '#button_type' => 'danger',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $uploads = $form_state->getValue('csv_file');
    $fid = $uploads[0];

    $batch_builder = (new BatchBuilder())
      ->setTitle($this->t('Importing CSV file'))
      ->setFinishCallback([AltTextImportBatch::class, 'finish'])
      ->setInitMessage($this->t('Starting import.'))
      ->setProgressMessage($this->t('Completed @current row of @total.'))
      ->setErrorMessage($this->t('CSV import has encountered an error.'));

    $batch_builder->addOperation([AltTextImportBatch::class, 'processRow'], [$fid]);

    batch_set($batch_builder->toArray());
  }

}
