<?php

namespace Drupal\alt_text_validation\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\State\State;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for Auditor to process and build the audit.
 */
class Auditor implements AuditorInterface, ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The queue name for the entity type queue.
   *
   * This is the primary queue that generates the secondary queue.
   */
  public const ATV_ENTITY_TYPE_QUEUE = 'atv_entity_types';

  /**
   * The queue name for the entity instances that have image capable fields.
   *
   * This is the secondary queue and contains an entry for each entity instance.
   */
  public const ATV_ENTITY_INSTANCE_QUEUE = 'atv_entity_instances';

  /**
   * The Alt Text Validation audit storage service.
   *
   * @var \Drupal\alt_text_validation\Service\AuditStorageInterface
   */
  public $auditStorage;

  /**
   * The Database connector service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $databaseConnection;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The alt_text_validation logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  public $logger;

  /**
   * The queue service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queue;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\State
   */
  public $state;

  /**
   * Constructs the AuditStorage service.
   *
   * @param Drupal\alt_text_validation\Service\AuditStorageInterface $audit_storage
   *   The audit storage service.
   * @param Drupal\Core\Database\Connection $database_connection
   *   The database connection service.
   * @param Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param Psr\Log\LoggerInterface $logger
   *   The alt_text_validation logger.
   * @param Drupal\Core\Queue\QueueFactory $queue
   *   The queue service.
   * @param Drupal\Core\State\State $state
   *   The state service.
   */
  final public function __construct(
    AuditStorageInterface $audit_storage,
    Connection $database_connection,
    EntityFieldManagerInterface $entity_field_manager,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerInterface $logger,
    QueueFactory $queue,
    State $state,
  ) {
    $this->auditStorage = $audit_storage;
    $this->databaseConnection = $database_connection;
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger;
    $this->queue = $queue;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('alt_text_validation.audit_storage'),
      $container->get('database'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('logger.channel.alt_text_validation'),
      $container->get('queue'),
      $container->get('state')
    );
  }

  /**
   * Collects all field instances that might contain alt text.
   *
   * @return array
   *   An array of fields that might contain alt text.
   */
  protected function collectImageContainingFields(): array {
    $image_instances = [];
    $entity_types = $this->entityFieldManager->getFieldMap();
    // Loop through retrieved field instances and access relevant information.
    foreach ($entity_types as $entity_type => $entity_fields) {
      foreach ($entity_fields as $entity_field => $field_info) {
        if ($this->isImageTypeField($field_info['type'])) {
          foreach ($field_info['bundles'] as $bundle) {
            $image_instances[$entity_type][$bundle][] = $entity_field;
          }
        }
      }
    }
    return $image_instances;
  }

  /**
   * {@inheritdoc}
   */
  public function queueAllImages(): void {
    // We want to reset the audit table and any items still in queues.
    $this->auditStorage->truncateTable();
    $this->truncateQueues();
    $states = [
      'alt_text_validation.audit_status' => $this->t('Queueing entities with images.'),
      'alt_text_validation.audit_start_time' => time(),
    ];
    $this->state->setMultiple($states);
    $image_containing_entities = $this->collectImageContainingFields();
    $atvqueue = $this->queue->get(self::ATV_ENTITY_TYPE_QUEUE);
    foreach ($image_containing_entities as $entity_type => $bundles) {
      foreach ($bundles as $bundle => $fields) {
        $item = new \stdClass();
        $item->entity_type = $entity_type;
        $item->bundle = $bundle;
        $item->fields = $fields;
        $atvqueue->createItem($item);
        $vars = [
          '@entity_type' => $entity_type,
          '@bundle' => $bundle,
        ];
        $this->logger->info('Queued entity type: @entity_type bundle: @bundle', $vars);
      }
    }
    $item_count = $atvqueue->numberOfItems();
    $vars = [
      '@queue_name' => self::ATV_ENTITY_TYPE_QUEUE,
      '@count' => $item_count,
    ];
    $this->logger->info('Queued @count entity:bundles that have fields that could support images into @queue_name.', $vars);
  }

  /**
   * Truncates all alt_text_validation queues.
   */
  protected function truncateQueues(): void {
    $this->queue->get(self::ATV_ENTITY_TYPE_QUEUE)->deleteQueue();
    $this->queue->get(self::ATV_ENTITY_INSTANCE_QUEUE)->deleteQueue();
  }

  /**
   * {@inheritdoc}
   */
  public function isImageTypeField(string $field_type): bool {
    return in_array($field_type, $this->getImageFieldTypes());
  }

  /**
   * {@inheritdoc}
   */
  public function getImageFieldTypes(): array {
    return ['image', 'text', 'text_long', 'text_with_summary'];
  }

}
