<?php

namespace Drupal\alt_text_validation\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for Validator to validate image alt, file and title.
 */
class Validator implements ValidatorInterface, ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The configuration for alt_text_validation.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $atvConfig;

  /**
   * The alt_text_validation logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs the AuditStorage service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param Psr\Log\LoggerInterface $logger
   *   The alt_text_validation logger.
   */
  final public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerInterface $logger,
  ) {
    $this->logger = $logger;
    $this->atvConfig = $config_factory->getEditable('alt_text_validation.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('logger.channel.alt_text_validation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getViolations(string $filename, string $alt, string $title): array {
    $violation_messages = [];
    // @todo add logic to make this work.
    // Load the rules.
    // Loop through applying the rules and build failure messages.
    return $violation_messages;
  }

  /**
   * {@inheritdoc}
   */
  public function getWarnings(string $filename, string $alt, string $title): array {
    $warning_messages = [];
    // @todo add logic to make this work.
    // Load the rules.
    // Loop through applying the rules and build failure messages.
    return $warning_messages;
  }

  /**
   * {@inheritdoc}
   */
  public function validateImageField(string $field_name, EntityInterface $entity): array {
    $validations = [];
    if ($this->isImageField($field_name, $entity)) {
      $field_values = $entity->get($field_name)->getValue();
      foreach ($field_values as $key => $field_value) {
        $file = $entity->get($field_name)?->referencedEntities()[$key];
        $file_uri = $file?->getFileUri();
        $file_source = $file->createFileUrl(FALSE);
        $validations[$file_source] = [
          'field_name' => $field_name,
          'file_name' => $file_name = pathinfo($file_uri, PATHINFO_BASENAME),
          'file_source' => $file_source,
          'title' => $title = $field_value['title'] ?? '',
          'alt' => $alt = $field_value['alt'] ?? '',
          'violations' => $this->getViolations($file_name, $alt, $title),
          'warnings' => $this->getWarnings($file_name, $alt, $title),
          'entity_type' => $entity->getEntityTypeId(),
          'bundle' => $entity->bundle(),
          'id' => $entity->id(),
        ];
      }
    }
    return $validations;
  }

  /**
   * {@inheritdoc}
   */
  public function validateTextField(string $field_name, EntityInterface $entity): array {
    $validations = [];
    // @todo add this logic
    // @see https://www.drupal.org/project/alt_text_validation/issues/3498668
    return $validations;
  }

  /**
   * {@inheritdoc}
   */
  public function isImageField(string $field_name, EntityInterface $entity):  bool {
    // A "thumbnail" is a secondary field of image field and should be ignored.
    return (
      $entity->getFieldDefinition($field_name)->getType() === 'image')
      && ($field_name !== 'thumbnail'
    );
  }

}
