<?php

namespace Drupal\alt_text_validation\Plugin\QueueWorker;

use Drupal\alt_text_validation\Service\Auditor;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerBase;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes queued entity bundles and queues images found.
 *
 * @QueueWorker(
 *   id = "atv_entity_types",
 *   title = @Translation("Queue entities instances for image field processing"),
 *   cron = {"time" = 10}
 * )
 */
class EntityImageFieldQueuer extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Auditor service.
   *
   * @var \Drupal\alt_text_validation\Service\Auditor
   */
  protected $auditor;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The alt_text_validation logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  public $logger;

  /**
   * The queue service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queue;

  /**
   * EntityImageFieldQueuer constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\alt_text_validation\Service\Auditor $auditor
   *   Auditor service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger_channel
   *   The alt_text_validation logger.
   * @param \Drupal\Core\Queue\QueueFactory $queue
   *   The queue service.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    Auditor $auditor,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerInterface $logger_channel,
    QueueFactory $queue,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
    $this->auditor = $auditor;
    $this->entityTypeManager = $entityTypeManager;
    $this->logger = $logger_channel;
    $this->queue = $queue;
  }

  /**
   * Inherited.
   *
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('alt_text_validation.auditor'),
      $container->get('entity_type.manager'),
      $container->get('logger.channel.alt_text_validation'),
      $container->get('queue'),
    );
  }

  /**
   * Inherited.
   *
   * {@inheritdoc}
   */
  public function processItem($data) {
    $entity_type = $data->entity_type;
    $bundle = $data->bundle;
    $fields = $data->fields;
    // Get all the published entities of the given type and bundle.
    // Queue the image fields for processing.
    $query = $this->entityTypeManager->getStorage($entity_type)->getQuery();
    // This is an audit of all image alts in the system, not just the ones a
    // specific role has access to. The audit report access is initially limited
    // to administrators.
    $query->accessCheck(FALSE);
    // Some entities have no bundles, and end up with the entity type as the
    // bundle. We don't want to filter by bundle in that case.
    if ($entity_type !== $bundle) {
      // Media entities, have bundle instead of type.
      $bundle_type = ($entity_type === 'media') ? 'bundle' : 'type';
      $bundle_type = ($entity_type === 'taxonomy_term') ? 'vid' : $bundle_type;
      $query->condition($bundle_type, $bundle);
    }

    if (in_array($entity_type, ['node', 'media', 'taxonomy_term'])) {
      // These are types that implement status.
      // @todo Detect this dynamically with by checking if the entity type
      // implements EntityPublishedInterface.
      $query->condition('status', 1);
    }
    $entity_ids = $query->execute();
    $start = time();
    foreach ($entity_ids as $revision => $id) {
      // Queue this because it could be a lot of items to process.
      $this->queue->get($this->auditor::getEntityInstanceQueueName())->createItem([
        'entity_type' => $entity_type,
        'bundle' => $bundle,
        'id' => $id,
        'fields' => $fields,
      ]);

    }

    $this->logger->info('Queued @count entities of type @type and bundle @bundle for image field auditing. Run time: @execution_time s.', [
      '@count' => count($entity_ids),
      '@type' => $entity_type,
      '@bundle' => $bundle,
      '@execution_time' => time() - $start,
    ]);
  }

}
