<?php

namespace Drupal\alt_text_validation\Plugin\QueueWorker;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\alt_text_validation\Service\AuditorInterface;
use Drupal\alt_text_validation\Service\ValidatorInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\State\State;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Audits the image carrying fields of queued entity instance.
 *
 * @QueueWorker(
 *   id = "atv_entity_instances",
 *   title = @Translation("Process the queued entity instances for image fields."),
 *   cron = {"time" = 45}
 * )
 */
class ImageFieldAuditor extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * Auditor service.
   *
   * @var \Drupal\alt_text_validation\Service\AuditorInterface
   */
  protected $auditor;

  /**
   * Validator service.
   *
   * @var \Drupal\alt_text_validation\Service\ValidatorInterface
   */
  protected $validator;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The queue service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queue;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\State
   */
  protected $state;

  /**
   * ImageFieldAuditor constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\alt_text_validation\Service\AuditorInterface $auditor
   *   Auditor service.
   * @param \Drupal\alt_text_validation\Service\ValidatorInterface $validator
   *   Validator service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param Drupal\Core\Queue\QueueFactory $queue
   *   The queue service.
   * @param \Drupal\Core\State\State $state
   *   The state service.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    AuditorInterface $auditor,
    ValidatorInterface $validator,
    EntityTypeManagerInterface $entity_type_manager,
    QueueFactory $queue,
    State $state,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
    $this->auditor = $auditor;
    $this->validator = $validator;
    $this->entityTypeManager = $entity_type_manager;
    $this->queue = $queue;
    $this->state = $state;
  }

  /**
   * Inherited.
   *
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('alt_text_validation.auditor'),
      $container->get('alt_text_validation.validator'),
      $container->get('entity_type.manager'),
      $container->get('queue'),
      $container->get('state')
    );
  }

  /**
   * Inherited.
   *
   * {@inheritdoc}
   */
  public function processItem($data) {
    $entity_type = $data['entity_type'];
    $bundle = $data['bundle'];
    $id = $data['id'];
    $fields = $data['fields'];
    $entity_instance = $this->entityTypeManager->getStorage($entity_type)->load($id);
    if (empty($entity_instance)) {
      // There is no entity to process, so bail out.
      return;
    }
    $validations = [];
    foreach ($fields as $field_name) {
      $image_field_validations = $this->validator->validateImageField($field_name, $entity_instance);
      $text_field_validations = $this->validator->validateTextField($field_name, $entity_instance);
      $validations = array_merge($validations, $image_field_validations, $text_field_validations);
    }

    foreach ($validations as $validation) {
      $this->auditor->auditStorage->writeAuditRow(
        $validation['file_name'],
        $validation['alt'],
        count($validation['warnings']),
        count($validation['violations']),
        $validation['title'],
        $validation['entity_type'],
        $validation['bundle'],
        $validation['id'],
        $validation['field_name'],
        $validation['file_source'],
        $validation['violations']
      );
    }
    // Check the queue count.  if empty, set the end time.
    if ($this->queue->get(AtvCommonTools::ENTITY_INSTANCE_QUEUE_NAME)->numberOfItems() <= 1) {
      $states = [
        AtvCommonTools::AUDIT_STATUS_KEY => $this->t('Completed'),
        AtvCommonTools::AUDIT_END_TIME_KEY => time(),
      ];
      $this->state->setMultiple($states);
    }
  }

}
