<?php

namespace Drupal\alt_text_validation\Service;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\State\State;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for Audit Storage to read and write entries to the audit table.
 */
class AuditStorage implements AuditStorageInterface, ContainerInjectionInterface {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The Database connector service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $databaseConnection;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The alt_text_validation logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  public $logger;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\State
   */
  protected $state;

  /**
   * Constructs the AuditStorage service.
   *
   * @param Drupal\Core\Database\Connection $database_connection
   *   The database connection service.
   * @param Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param Psr\Log\LoggerInterface $logger_channel
   *   The alt_text_validation logger channel.
   * @param Drupal\Core\State\State $state
   *   The state service.
   */
  final public function __construct(
    Connection $database_connection,
    EntityFieldManagerInterface $entity_field_manager,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerInterface $logger_channel,
    State $state,
  ) {
    $this->databaseConnection = $database_connection;
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_channel;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('logger.channel.alt_text_validation'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function writeAuditRow(
    string $filename,
    string $alt_text,
    int $warnings,
    int $violations,
    string $title,
    string $entity,
    string $bundle,
    int $id,
    string $field_name,
    string $file_source,
    array $violation_list,
  ): void {

    $result = $this->databaseConnection->insert(AtvCommonTools::AUDIT_TABLE_NAME)
      ->fields([
        'filename' => $filename,
        'alt_text' => $alt_text,
        'warnings' => $warnings,
        'violations' => $violations,
        'title' => $title,
        'entity' => $entity,
        'bundle' => $bundle,
        'id' => $id,
        'field_name' => $field_name,
        'file_source' => $file_source,
        'violation_list' => $this->formatList($violation_list),
      ])
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function truncateTable(): void {
    $this->databaseConnection->truncate(AtvCommonTools::AUDIT_TABLE_NAME)->execute();
    $states = [
      AtvCommonTools::AUDIT_STATUS_KEY => '',
      AtvCommonTools::AUDIT_START_TIME_KEY => NULL,
      AtvCommonTools::AUDIT_END_TIME_KEY => NULL,
    ];
    $this->state->setMultiple($states);
  }

  /**
   * Formats an array into an html list.
   *
   * @param array $list
   *   An array of items to turn into a list.
   *
   * @return string
   *   An html string containing the list.
   */
  protected function formatList(array $list): string {
    $formatted_list = '';
    if (!empty($list)) {
      $formatted_list .= '<ul>';
      foreach ($list as $item) {
        $formatted_list .= "<li>{$item}</li>";
      }
      $formatted_list .= '</ul>';
    }

    return $formatted_list;

  }

  /**
   * {@inheritdoc}
   *
   * @todo move this to the unit test when there is one.
   */
  public function generateTestData(): void {
    $test_rows = [
      [
        'filename' => 'juniper.jpg',
        'alt_text' => 'Juniper berries',
        'warnings' => 0,
        'violations' => 0,
        'title' => 'The berries of the Juniper tree',
        'entity' => 'node',
        'bundle' => 'news',
        'id' => 7,
        'field_name' => 'news_cover',
        'file_source' => '/sites/default/trees/juniper.jpg',
        'validation_list' => [],
      ],
      [
        'filename' => 'cypress.jpg',
        'alt_text' => 'cypress.jpg',
        'warnings' => 0,
        'violations' => 1,
        'title' => 'cypress.jpg',
        'entity' => 'taxonomy_term',
        'bundle' => 'trees',
        'id' => '531',
        'field_name' => 'thumbnail',
        'file_source' => '/sites/default/trees/cypress.jpg',
        'validation_list' => [
          'Alt text should not contain the filename.  That is annoying to listen to.',
          'The alt and title attributes must not duplicate each other.',
        ],
      ],
      [
        'filename' => 'pine.jpg',
        'alt_text' => '',
        'warnings' => 0,
        'violations' => 1,
        'title' => '',
        'entity' => 'taxonomy_term',
        'bundle' => 'trees',
        'id' => 61,
        'field_name' => 'thumbnail',
        'file_source' => '/sites/default/trees/pine.jpg',
        'validation_list' => ['Alt text should not be empty.'],
      ],
      [
        'filename' => 'oak.gif',
        'alt_text' => 'Drawing of an acorn.',
        'warnings' => 1,
        'violations' => 0,
        'title' => 'Drawing of an acorn.',
        'entity' => 'media',
        'bundle' => 'images',
        'id' => 501,
        'field_name' => 'image',
        'file_source' => '/sites/default/files/images/oak.gif',
        'validation_list' => [],
      ],
    ];
    $this->truncateTable();
    foreach ($test_rows as $row) {
      $this->writeAuditRow(
        $row['filename'],
        $row['alt_text'],
        $row['warnings'],
        $row['violations'],
        $row['title'],
        $row['entity'],
        $row['bundle'],
        $row['id'],
        $row['field_name'],
        $row['file_source'],
        $row['validation_list']
      );
    }
  }

}
