<?php

namespace Drupal\alt_text_validation\Service;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for Validator to validate image alt, file and title.
 */
class Validator implements ValidatorInterface, ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The configuration for alt_text_validation.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $atvConfig;

  /**
   * The alt_text_validation logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs the AuditStorage service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param Psr\Log\LoggerInterface $logger
   *   The alt_text_validation logger.
   */
  final public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerInterface $logger,
  ) {
    $this->logger = $logger;
    $this->atvConfig = $config_factory->getEditable('alt_text_validation.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('logger.channel.alt_text_validation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getViolations(string $filename, string $alt, string $title): array {
    $violation_messages = [];
    // @todo add logic to make this work.
    // Load the rules.
    // Loop through applying the rules and build failure messages.
    return $violation_messages;
  }

  /**
   * {@inheritdoc}
   */
  public function getWarnings(string $filename, string $alt, string $title): array {
    $warning_messages = [];
    // @todo add logic to make this work.
    // Load the rules.
    // Loop through applying the rules and build failure messages.
    return $warning_messages;
  }

  /**
   * {@inheritdoc}
   */
  public function validateImageField(string $field_name, EntityInterface $entity): array {
    $validations = [];
    if (AtvCommonTools::isImageField($field_name, $entity)) {
      $field_values = $entity->get($field_name)->getValue();
      foreach ($field_values as $key => $field_value) {
        $file = $entity->get($field_name)?->referencedEntities()[$key];
        $file_uri = $file?->getFileUri();
        $file_source = $file->createFileUrl(FALSE);
        $validations[$file_source] = [
          'field_name' => $field_name,
          'file_name' => $file_name = pathinfo($file_uri, PATHINFO_BASENAME),
          'file_source' => $file_source,
          'title' => $title = $field_value['title'] ?? '',
          'alt' => $alt = $field_value['alt'] ?? '',
          'violations' => $this->getViolations($file_name, $alt, $title),
          'warnings' => $this->getWarnings($file_name, $alt, $title),
          'entity_type' => $entity->getEntityTypeId(),
          'bundle' => $entity->bundle(),
          'id' => $entity->id(),
        ];
      }
    }
    return $validations;
  }

  /**
   * {@inheritdoc}
   */
  public function validateTextField(string $field_name, EntityInterface $entity): array {
    $validations = [];
    if (AtvCommonTools::isTextImageField($field_name, $entity)) {
      $field_values = $entity->get($field_name)->getValue();
      foreach ($field_values as $key => $field_value) {
        $html_string = $field_value['value'] ?? '';
        $images = $this->extractImageTags($html_string);
        foreach ($images as $image) {
          $file_source = $image['src'];
          $file_name = pathinfo($file_source, PATHINFO_BASENAME);
          $validations[] = [
            'field_name' => $field_name,
            'file_name' => $file_name,
            'file_source' => $file_source,
            'title' => $image['title'],
            'alt' => $image['alt'],
            'violations' => $this->getViolations($file_name, $image['alt'], $image['title']),
            'warnings' => $this->getWarnings($file_name, $image['alt'], $image['title']),
            'entity_type' => $entity->getEntityTypeId(),
            'bundle' => $entity->bundle(),
            'id' => $entity->id(),
          ];
        }
      }
    }

    return $validations;
  }

  /**
   * Extract images from a string of HTML.
   *
   * @param string $html_string
   *   The sting to extract images from.
   *
   * @return array
   *   Array of arrays each containing src, alt, and title elements.
   */
  protected function extractImageTags(string $html_string) {
    $images = [];
    if (empty($html_string)) {
      return $images;
    }
    // We are only loading a snippet of html so we need to wrap it to make it
    // look like a complete page.
    $html_string = "<!DOCTYPE html><html><body>$html_string</body></html>";
    $dom = new \DOMDocument();
    // The contents of the text field may trigger some DomDocument warnings
    // that we don't need to see. So we silence them temporarily.
    libxml_use_internal_errors(TRUE);
    $dom->loadHtml($html_string);
    libxml_use_internal_errors(FALSE);
    foreach ($dom->getElementsByTagName('img') as $element) {
      $images[] = [
        'src' => $element->getAttribute('src'),
        'alt' => $element->getAttribute('alt'),
        'title' => $element->getAttribute('title'),
      ];
    }
    return $images;
  }

}
