<?php

namespace Drupal\alt_text_validation;

use Drupal\Core\Entity\EntityInterface;

/**
 * Common methods and constants.
 */
class AtvCommonTools {

  /**
   * The name of the audit table.
   *
   * @var string
   */
  public const AUDIT_TABLE_NAME = 'alt_text_validation_audit';

  /**
   * The State key for storing the audit status.
   *
   * @var string
   */
  public const AUDIT_STATUS_KEY = 'alt_text_validation.audit_status';

  /**
   * The State key for storing the audit start time.
   *
   * @var string
   */
  public const AUDIT_START_TIME_KEY = 'alt_text_validation.audit_start_time';

  /**
   * The State key for storing the audit end time.
   *
   * @var string
   */
  public const AUDIT_END_TIME_KEY = 'alt_text_validation.audit_end_time';

  /**
   * The queue name for the entity type queue.
   *
   * @var string
   *
   * This is the primary queue that generates the secondary queue.
   */
  public const ENTITY_TYPE_QUEUE_NAME = 'atv_entity_types';

  /**
   * The queue name for the entity type queue.
   *
   * @var string
   *
   * This is the primary queue that generates the secondary queue.
   */
  public const ENTITY_INSTANCE_QUEUE_NAME = 'atv_entity_instances';

  /**
   * Checks if a field is considered an image field.
   *
   * @param string $field_name
   *   The field name.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return bool
   *   TRUE if the field is an image field, FALSE otherwise.
   */
  public static function isImageField(string $field_name, EntityInterface $entity): bool {
    // A "thumbnail" is a sub-field of image field and should be ignored.
    return (
      in_array($entity->getFieldDefinition($field_name)->getType(), self::getImageFieldTypes())
      && ($field_name !== 'thumbnail')
    );
  }

  /**
   * Checks if a field is considered a text field that can contain image tags.
   *
   * @param string $field_name
   *   The field name.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return bool
   *   TRUE if the field is an image field, FALSE otherwise.
   */
  public static function isTextImageField(string $field_name, EntityInterface $entity): bool {
    return AtvCommonTools::isTextImageFieldType($entity->getFieldDefinition($field_name)->getType());
  }

  /**
   * Gets the type of fields that are image fields.
   *
   * @return array
   *   An array of field types considered to be image fields.
   */
  public static function getImageFieldTypes(): array {
    // At the moment we are only considering only a true image field.
    return ['image'];
  }

  /**
   * Checks to see if the field type is an alt containing field type.
   *
   * @param string $field_type
   *   The field type to check.
   *
   * @return bool
   *   TRUE if the field type is an image type field, FALSE otherwise.
   */
  public static function isAltContainingTypeField(string $field_type): bool {
    return in_array($field_type, AtvCommonTools::getAltContainingFieldTypes());
  }

  /**
   * Gets the type of fields that can have alt text.
   *
   * @return array
   *   An array of field types that can have alt text.
   */
  public static function getAltContainingFieldTypes(): array {
    return array_merge(AtvCommonTools::getImageFieldTypes(), AtvCommonTools::getTextImageFieldTypes());
  }

  /**
   * Gets the type of fields that can have images in the text.
   *
   * @return array
   *   An array of field types that can have images in the text.
   */
  public static function getTextImageFieldTypes(): array {
    return ['text', 'text_long', 'text_with_summary'];
  }

  /**
   * Checks if a field is considered a text or image field.
   *
   * @param string $field_type
   *   The field type.
   *
   * @return bool
   *   TRUE if the field is text field that can contain images, FALSE otherwise.
   */
  public static function isTextImageFieldType(string $field_type): bool {
    return in_array($field_type, AtvCommonTools::getTextImageFieldTypes());
  }

}
