<?php

namespace Drupal\alt_text_validation\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the Text Rule add and edit forms.
 */
class AltTextRuleForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\alt_text_validation\AltTextRuleInterface $entity */
    $entity = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#description' => $this->t('The text rule label. This is used to identify the rule in the Alt Text Report.'),
      '#default_value' => $entity->get('label'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->get('id'),
      '#machine_name' => [
        'exists' => [$this, 'exists'],
        'source' => ['label'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $preamble = $this->t("Choose how the rule should behave when it finds alt text.");
    $prevent = $this->t("<strong>Prevent</strong> won't allow users to save if the alt text does not meet the rule.");
    $warn = $this->t("<strong>Warn</strong> allows users to save but displays a warning encouraging them to edit their alt text.");
    $off = $this->t("<strong>Off</strong> ignores this rule.");
    $form['violation_action'] = [
      '#type' => 'select',
      '#title' => $this->t('Rule status'),
      '#description' => "$preamble </br> $off </br> $warn </br> $prevent",
      '#default_value' => $entity->get('violation_action') ?? '',
      '#empty_option' => $this->t('- select -'),
      '#empty_value' => '',
      '#options' => [
        'off' => $this->t('Off'),
        'warn' => $this->t('Warn'),
        'prevent' => $this->t('Prevent'),
      ],
      '#required' => TRUE,
    ];

    $form['rule_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Rule validation'),
      '#description' => $this->t('What you want to validate the alt text against.'),
      '#default_value' => $entity->getRuleType() ?? '',
      '#empty_option' => $this->t('- select -'),
      '#empty_value' => '',
      '#options' => $entity->getRuleTypeOptions(),
      '#required' => TRUE,
    ];

    $form['content_char_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Content character limit'),
      '#default_value' => $entity->get('content_char_limit'),
      '#description' => $this->t('The number of characters allowed.'),
      '#min' => 0,
      '#states' => [
        'visible' => [
          ':input[name="rule_type"]' => ['value' => 'length_limit'],
        ],
        'required' => [
          ':input[name="rule_type"]' => ['value' => 'length_limit'],
        ],
      ],
    ];

    $form['text_to_compare'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Text to compare'),
      '#default_value' => $entity->get('text_to_compare'),
      '#description' => $this->t("The text you want to check against the submitted alt text."),
      '#states' => [
        'visible' => [
          ':input[name="rule_type"]' => [
            ['value' => 'not_begin_with'],
            ['value' => 'not_equal'],
            ['value' => 'not_contain'],
            ['value' => 'not_end_with'],
            ['value' => 'regex_match'],
          ],
        ],
        'required' => [
          ':input[name="rule_type"]' => [
            ['value' => 'not_begin_with'],
            ['value' => 'not_equal'],
            ['value' => 'not_contain'],
            ['value' => 'not_end_with'],
            ['value' => 'regex_match'],
          ],
        ],
      ],
    ];

    $form['violation_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Violation Message'),
      '#default_value' => $entity->get('violation_message') ?? '',
      '#description' => $this->t("The error message that displays when the rule is broken. Tell your users what the issue is and how to fix it. For example 'This alt text is too long. Please shorten your alt text to 100 characters or less.'"),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $alt_text_rule = $this->entity;
    $status = $alt_text_rule->save();

    if ($status) {
      $this->messenger()->addMessage($this->t('Saved the %label Text Rule.', [
        '%label' => $alt_text_rule->label(),
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('Failed to save the %label Text Rule.', [
        '%label' => $alt_text_rule->label(),
      ]), 'error');
    }

    $form_state->setRedirect('entity.alt_text_rule.collection');
  }

  /**
   * Helper function to check whether a Text Rule config entity exists.
   *
   * @returns bool
   */
  public function exists($id): bool {
    $entity = $this->entityTypeManager
      ->getStorage('alt_text_rule')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('id', $id)
      ->execute();
    return (bool) $entity;
  }

  /**
   * Constructs a TextRule config entity.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static($container->get('entity_type.manager'));
  }

}
