<?php

namespace Drupal\Tests\alt_text_validation\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\alt_text_validation\Entity\AltTextRule;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the AltTextRule::isViolation method for different rule types.
 *
 * @group alt_text_validation
 * @coversDefaultClass \Drupal\alt_text_validation\Entity\AltTextRule
 */
class RuleIsViolationTest extends UnitTestCase {

  /**
   * Tests alt_is_filename rule type.
   */
  public function testAltIsFilename() {
    $rule = $this->mockRuleOfType('alt_is_filename');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'image.jpg', 'Some Title'), 'The alt was the filename, should have been a violation.');

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Good alt text', 'Some Title'), 'The alt was not the filename, should not have been a violation.');
  }

  /**
   * Tests alt_is_title rule type.
   */
  public function testAltIsTitle() {
    $rule = $this->mockRuleOfType('alt_is_title');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'My Title', 'My Title'), 'The alt text was the same as the title, should have been a violation.');

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Good alt text', 'My Title'), 'The alt text was not the same as the title, should not have been a violation.');
  }

  /**
   * Tests not_empty rule type.
   */
  public function testNotEmpty() {
    $rule = $this->mockRuleOfType('not_empty');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', '', 'My Title'));

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Good alt text', 'My Title'));
  }

  /**
   * Tests length_limit rule type.
   */
  public function testLengthLimit() {
    $rule = $this->mockRuleOfType('length_limit');
    $rule->method('getContentCharLimit')->willReturn(10);
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'This text is too long', 'My Title'));

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Short', 'My Title'));
  }

  /**
   * Tests not_begin_with rule type.
   */
  public function testNotBeginWith() {
    $rule = $this->mockRuleOfType('not_begin_with');
    $rule->method('getTextToCompare')->willReturn('Image of');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'Image of a cat', 'My Title'));

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Cat playing', 'My Title'));
  }

  /**
   * Tests not_equal rule type.
   */
  public function testNotEqual() {
    $rule = $this->mockRuleOfType('not_equal');
    $rule->method('getTextToCompare')->willReturn('blood');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'blood', 'My Title'));

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Good alt text', 'My Title'));
  }

  /**
   * Tests not_contain rule type.
   */
  public function testNotContain() {
    $rule = $this->mockRuleOfType('not_contain');
    $rule->method('getTextToCompare')->willReturn('placeholder');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'This is a placeholder image', 'My Title'));

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Good alt text', 'My Title'));
  }

  /**
   * Tests not_end_with rule type.
   */
  public function testNotEndWith() {
    $rule = $this->mockRuleOfType('not_end_with');
    $rule->method('getTextToCompare')->willReturn('The end.');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'My life was good. The end.', 'My Title'));

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'Good alt text', 'image.jpg', 'My Title'));
  }

  /**
   * Tests regex_match rule type.
   */
  public function testRegexMatch() {
    $rule = $this->mockRuleOfType('regex_match');
    $rule->method('getTextToCompare')->willReturn('/^img_\d+$/');
    $method = $this->getMethodIsViolation();

    // Test violation case.
    $this->assertTrue($method->invoke($rule, 'image.jpg', 'img_123', 'My Title'));

    // Test non-violation case.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Good alt text', 'My Title'));
  }

  /**
   * Tests default case.
   */
  public function testDefaultCase() {
    $rule = $this->mockRuleOfType('unknown_rule_type');
    $method = $this->getMethodIsViolation();

    // Test default case of no type should return false.
    $this->assertFalse($method->invoke($rule, 'image.jpg', 'Any alt text', 'My Title'));
  }

  /**
   * Creates a mock of AltTextRule with a specific rule type.
   *
   * @param string $rule_type
   *   The rule type to mock.
   *
   * @return \PHPUnit\Framework\MockObject\MockObject
   *   The mocked AltTextRule object.
   */
  protected function mockRuleOfType(string $rule_type): MockObject {
    $rule = $this->createMock(AltTextRule::class);
    $rule->method('getRuleType')
      ->willReturn($rule_type);
    return $rule;
  }

  /**
   * Gets the isViolation method of AltTextRule for testing.
   *
   * @return mixed
   *   The reflected isViolation method of AltTextRule.
   */
  protected function getMethodIsViolation() {
    $method = new \ReflectionMethod(AltTextRule::class, 'isViolation');
    $method->setAccessible(TRUE);
    return $method;
  }

}
