/**
 * @file
 * Provides the preview functionality for the Alternative Color module. This
 * makes the live preview of the theme work.
 */

((Drupal, settings, once) => {
  /**
   * Updates the preview iFrame with the new color values.
   *
   * @param {Array} colors
   *   An array of color hex values keyed by their field name.
   */
  function updatePreview(colors) {
    let style = '';

    Object.entries(colors).forEach(([key, value]) => {
      style += `--color-${key}: ${value};`;
    });

    // Define the preview iFrame object.
    const iFrameBody = window.frames['altcolor-preview-frame'].document;
    iFrameBody.querySelector('html').setAttribute('style', style);
  }

  /**
   * Event handler for updating the preview.
   *
   * @param {CustomEvent} event
   *   The altcolor-change event.
   */
  function OnAltColorChanged(event) {
    updatePreview(event.detail.colors);
  }

  /**
   * Event handler that executes when the user navigates the iframe. This
   * enables the color overrides to be re-applied when the user is navigating
   * through the website in the iframe.
   */
  function OnIframeLoad() {
    // Update the preview with the current color values.
    updatePreview(Drupal.altcolor.getColors());
  }

  /**
   * Initializes the Alternative Color preview.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the behavior for Alternative Color.
   */
  Drupal.behaviors.altcolorPreview = {
    attach: (context) => {
      // Listen to the altcolor-change event on the form object. This custom
      // event is defined and triggered in altcolor_admin.js.
      once('altcolor-form', '#altcolor_form', context).forEach((form) => {
        form.addEventListener('altcolor-change', OnAltColorChanged);
      });

      // Attach an event listener to the iframe so that it updates the color
      // preview when the page loads. This includes navigation to other pages.
      once(
        'altcolor-preview-frame',
        'iframe[name="altcolor-preview-frame"]',
        context,
      ).forEach((iframe) => {
        iframe.addEventListener('load', OnIframeLoad);
      });
    },
  };
})(Drupal, drupalSettings, once);
