<?php

namespace Drupal\altcolor\Hook;

use Drupal\altcolor\Plugin\AltColorPluginManagerInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Theme\ThemeManagerInterface;

/**
 * Theme pre-processing hooks for the Alternative color module.
 */
class AltColorPreprocessHooks {

  /**
   * Constructor for AltColorPreprocessHooks.
   *
   * @param \Drupal\altcolor\Plugin\AltColorPluginManagerInterface $altColorManager
   *   The Alternative color plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $themeManager
   *   The theme manager.
   */
  public function __construct(
    protected AltColorPluginManagerInterface $altColorManager,
    protected ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
    protected ThemeManagerInterface $themeManager,
  ) {
  }

  /**
   * Implements hook_preprocess_HOOK() for HTML renderings.
   *
   * Here the color variables are injected into the page's HTML element style
   * attribute. There, they take precedence over the values that were originally
   * set in the theme.
   */
  #[Hook('preprocess_html')]
  public function preprocessHtml(&$variables): void {
    // Get the active theme.
    $theme = $this->themeManager->getActiveTheme()->getName();
    // See if the theme defines overridable colors.
    $theme_colors_definition = $this->altColorManager->getColorDefinitionsForActiveTheme();

    // If the theme provides color overrides...
    if (!empty($theme_colors_definition) && $theme_colors_definition->getColors()) {
      // Get the colors from the theme configuration. Use an empty array when
      // undefined.
      $colors = theme_get_setting('third_party_settings.altcolor.colors', $theme) ?? [];

      // Extract the cacheable metadata from the variables.
      $cacheable_metadata = CacheableMetadata::createFromRenderArray($variables);
      // Colors are theme-dependent, always add theme cache context.
      $cacheable_metadata->addCacheContexts(['theme']);
      // Add the theme config as a dependency.
      $theme_config = $this->configFactory->get($theme . '.config');
      $cacheable_metadata->addCacheableDependency($theme_config);

      // Allow modules and themes to insert or overwrite derived colors.
      $array = [$theme, &$colors, &$cacheable_metadata];
      $this->moduleHandler->invokeAll('altcolor_alter_colors', $array);

      // Construct the CSS variable string.
      // @todo Consider adding to existing style attribute.
      // $style = $variables['html_attributes']->getAttribute('style') ?? '';
      $style = '';
      foreach ($colors as $variable => $value) {
        $style .= "--color-$variable: $value;";
      }
      str_replace('_', '-', $style);

      // Add the cacheable metadata back to the variables.
      $cacheable_metadata->applyTo($variables);

      // Set the style attribute on the HTML element.
      $variables['html_attributes']->setAttribute('style', $style);
    }
  }

}
