<?php

declare(strict_types=1);

namespace Drupal\Tests\altcolor\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the functionality of the alternative color module.
 *
 * @group altcolor
 */
class AlternativeColorTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['altcolor'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Enable the test themes.
    \Drupal::service('theme_installer')->install([
      'test_theme_altcolor1',
      'test_theme_altcolor2',
      'test_theme_altcolor_extended',
      'test_theme_altcolor_unsupported',
    ]);

    // Create and log in a user with administrative privileges.
    $admin = $this->drupalCreateUser(['administer themes']);
    $this->drupalLogin($admin);
  }

  /**
   * Tests the CSS variable injection.
   */
  public function testCssVariableInjection(): void {
    // Set the default theme to the first theme.
    $this->config('system.theme')->set('default', 'test_theme_altcolor1')->save();

    // Visit the front page.
    $this->drupalGet('');
    // Assert that there are no color overrides in the HTML style attribute.
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-base');
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-text');
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-accent');

    // @todo Assert that the elements in the page have the colors as defined in
    //   the theme CSS file?

    // Set some color values for the first theme.
    $color_base = '#f0f8ff';
    $color_text = '#1e3a5f';
    $color_accent = '#4a90b8';

    $this->config('test_theme_altcolor1.settings')
      ->set('third_party_settings.altcolor.colors', [
        'base' => $color_base,
        'text' => $color_text,
        'accent' => $color_accent,
      ])
      ->save(TRUE);

    // Visit the front page.
    $this->drupalGet('');
    // Assert that the color values are injected into the page.
    $this->assertSession()->elementAttributeContains('css', 'html', 'style', '--color-base: ' . $color_base);
    $this->assertSession()->elementAttributeContains('css', 'html', 'style', '--color-text: ' . $color_text);
    $this->assertSession()->elementAttributeContains('css', 'html', 'style', '--color-accent: ' . $color_accent);

    // Change the default theme to the second theme.
    $this->config('system.theme')->set('default', 'test_theme_altcolor2')->save();

    // Visit the front page.
    $this->drupalGet('');
    // Assert that there are no color overrides in the HTML style attribute from
    // the first theme nor the second theme.
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-base');
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-text');
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-accent');
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-background');
    $this->assertSession()->elementAttributeNotContains('css', 'html', 'style', '--color-foreground');

    // Set the default theme to the extended theme.
    $this->config('system.theme')->set('default', 'test_theme_altcolor_extended')->save();

    // Set some color values for the extended theme.
    $color_base = '#f8fcff';
    $color_text = '#0f2027';
    $color_accent = '#2c7da0';

    $this->config('test_theme_altcolor_extended.settings')
      ->set('third_party_settings.altcolor.colors', [
        'base' => $color_base,
        'text' => $color_text,
        'accent' => $color_accent,
      ])
      ->save(TRUE);

    // Visit the front page.
    $this->drupalGet('');
    // Assert that the color values are injected into the page.
    $this->assertSession()->elementAttributeContains('css', 'html', 'style', '--color-base: ' . $color_base);
    $this->assertSession()->elementAttributeContains('css', 'html', 'style', '--color-text: ' . $color_text);
    $this->assertSession()->elementAttributeContains('css', 'html', 'style', '--color-accent: ' . $color_accent);
  }

  /**
   * Tests the working of the theme settings form with alternative color.
   */
  public function testThemeSettingsForm(): void {
    // Assert that the color scheme form is not shown for unsupported themes.
    $this->drupalGet('admin/appearance/settings/test_theme_altcolor_unsupported');
    $this->assertSession()->fieldNotExists('altcolor[colors][base]');
    $this->assertSession()->fieldNotExists('altcolor[colors][text]');
    $this->assertSession()->fieldNotExists('altcolor[colors][accent]');

    // Visit a supported theme settings page.
    $this->drupalGet('admin/appearance/settings/test_theme_altcolor1');
    // Assert that the fields are shown.
    $this->assertSession()->fieldExists('altcolor[colors][base]');
    $this->assertSession()->fieldExists('altcolor[colors][text]');
    $this->assertSession()->fieldExists('altcolor[colors][accent]');

    // Save some new values for the fields.
    $edit = [
      'altcolor[colors][base]' => '#ffffaa',
      'altcolor[colors][text]' => '#ffffbb',
      'altcolor[colors][accent]' => '#ffffcc',
    ];
    $this->submitForm($edit, 'Save configuration');

    $this->drupalGet('admin/appearance/settings/test_theme_altcolor1');
    $this->assertSession()->statusCodeEquals(200);
    // Assert that the saved values are displayed in the form.
    $this->assertSession()->fieldValueEquals('altcolor[colors][base]', '#ffffaa');
    $this->assertSession()->fieldValueEquals('altcolor[colors][text]', '#ffffbb');
    $this->assertSession()->fieldValueEquals('altcolor[colors][accent]', '#ffffcc');

    // Visit another supported theme settings page.
    $this->drupalGet('admin/appearance/settings/test_theme_altcolor2');
    // Assert that the fields are shown.
    $this->assertSession()->fieldExists('altcolor[colors][background]');
    $this->assertSession()->fieldExists('altcolor[colors][foreground]');
    // Assert that the previously saved values are not displayed for this theme.
    // The theme should show the values from the first color scheme.
    $this->assertSession()->fieldValueEquals('altcolor[colors][background]', '#f0f8ff');
    $this->assertSession()->fieldValueEquals('altcolor[colors][foreground]', '#1e3a5f');
  }

  /**
   * Tests that color definition inheritance also works for a subtheme.
   */
  public function testThemeSettingsFormInheritedColorDefinition(): void {
    $this->drupalGet('admin/appearance/settings/test_theme_altcolor_extended');

    // Assert that the fields are shown.
    $this->assertSession()->fieldExists('altcolor[colors][background]');
    $this->assertSession()->fieldExists('altcolor[colors][foreground]');
  }

}
