/**
 * @file
 * Tests of the Alternative Color module scheme drop down functionality.
 */

const selectors = {
  schemePicker: '#edit-scheme',
  colorFields: {
    base: 'input[type="color"][name="altcolor[colors][base]"]',
    text: 'input[type="color"][name="altcolor[colors][text]"]',
    accent: 'input[type="color"][name="altcolor[colors][accent]"]',
  },
};

const colorSchemes = {
  aqua_depths: {
    base: '#f0f8ff',
    text: '#1e3a5f',
    accent: '#4a90b8',
  },
  arctic_stream: {
    base: '#f8fcff',
    text: '#0f2027',
    accent: '#2c7da0',
  },
};

module.exports = {
  '@tags': ['altcolor'],
  before(browser) {
    browser
      .drupalInstall()
      .drupalInstallModule('altcolor')
      .drupalEnableTheme('test_theme_altcolor1');
  },
  after(browser) {
    browser.drupalUninstall();
  },
  'Theme settings - Color schemes update individual values': (browser) => {
    browser.drupalLoginAsAdmin(() => {
      browser
        .drupalRelativeURL('/admin/appearance/settings/test_theme_altcolor1')
        .waitForElementVisible(selectors.schemePicker)
        // Assert that the first color scheme is selected by default.
        .assert.valueEquals(
          selectors.colorFields.base,
          colorSchemes.aqua_depths.base,
        )
        .assert.valueEquals(
          selectors.colorFields.text,
          colorSchemes.aqua_depths.text,
        )
        .assert.valueEquals(
          selectors.colorFields.accent,
          colorSchemes.aqua_depths.accent,
        )
        // Switch to the second color scheme.
        .click(`${selectors.schemePicker} option[value="arctic_stream"]`)
        .assert.valueEquals(
          selectors.colorFields.base,
          colorSchemes.arctic_stream.base,
        )
        .assert.valueEquals(
          selectors.colorFields.text,
          colorSchemes.arctic_stream.text,
        )
        .assert.valueEquals(
          selectors.colorFields.accent,
          colorSchemes.arctic_stream.accent,
        )
        // Switch back to the first color scheme.
        .click(`${selectors.schemePicker} option[value="aqua_depths"]`)
        .assert.valueEquals(
          selectors.colorFields.base,
          colorSchemes.aqua_depths.base,
        )
        .assert.valueEquals(
          selectors.colorFields.text,
          colorSchemes.aqua_depths.text,
        )
        .assert.valueEquals(
          selectors.colorFields.accent,
          colorSchemes.aqua_depths.accent,
        )
        // Test that switching to custom does not change the values.
        .click(`${selectors.schemePicker} option[value=""]`)
        .assert.valueEquals(
          selectors.colorFields.base,
          colorSchemes.aqua_depths.base,
        )
        .assert.valueEquals(
          selectors.colorFields.text,
          colorSchemes.aqua_depths.text,
        )
        .assert.valueEquals(
          selectors.colorFields.accent,
          colorSchemes.aqua_depths.accent,
        );
    });
  },
  'Theme settings - Changing value unsets scheme': (browser) => {
    browser.drupalLoginAsAdmin(() => {
      browser
        .drupalRelativeURL('/admin/appearance/settings/test_theme_altcolor1')
        .waitForElementVisible(selectors.schemePicker)
        .waitForElementVisible(selectors.colorFields.accent)
        .assert.valueEquals(selectors.schemePicker, 'aqua_depths');
      // The setValue function does not work with color fields. Set it manually.
      browser
        .execute(
          function (selector, value) {
            const element = document.querySelector(selector);
            element.value = value;
            element.dispatchEvent(new Event('input', { bubbles: true }));
            element.dispatchEvent(new Event('change', { bubbles: true }));
          },
          [selectors.colorFields.accent, '#009cde'],
        )
        .pause(1000)
        .waitForElementVisible(selectors.schemePicker)
        .assert.valueEquals(selectors.schemePicker, '');
    });
  },
};
