<?php

namespace Drupal\amazon_image_id_scan\Services;

use Aws\Command;
use Aws\Rekognition\Exception\RekognitionException;
use Aws\Rekognition\RekognitionClient;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Clase para la conexion a servicios de amazon.
 */
class Rekognition {
  /**
   * Variable para traer configuraciones.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * Constructor del formulario.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->config = $config_factory->get('amazon_image_id_scan.settings');
  }

  /**
   * Conexion con amazon.
   */
  public function connect() {
    $credentials = $this->config->get('rekognition_tab');
    $client_config['credentials'] = [
      'key' => $credentials['key'] ?? FALSE,
      'secret' => $credentials['secret'] ?? FALSE,
    ];
    $client_config['region'] = 'us-east-1';
    $client_config['version'] = 'latest';

    $this->connect = new RekognitionClient($client_config);
    return $this;
  }

  /**
   * Envio de archivo.
   */
  public function selfie($image, $config) {
    $return = FALSE;
    if (!empty($image)) {
      $this->connect();
      $command = new Command('Error');
      try {
        $result = $this->connect->detectLabels([
          'Image' => [
            'Bytes' => $image,
          ],
          'Attributes' => ['ALL'],
        ]);
        // Condiciones que no debe tener.
        $flag_neg = TRUE;
        $flag_pos = TRUE;
        $percents_arr = [];
        $messages_arr = [];
        $message_err = "";
        unset($config['container_negativos']['add']);
        if (isset($config['container_negativos'])
          && !empty($config['container_negativos'])) {
          $message_err = "No reconocemos tu rostro. Por favor ubícate frente a tu celular, acerca más tu cara y tómate una foto.";
          foreach ($result['Labels'] as $label) {
            foreach ($config['container_negativos'] as $label_pos) {
              if (isset($label_pos['label']) && $label['Name'] == $label_pos['label']) {
                $flag_neg = FALSE;
                $msg_to_arr = $label_pos['error_description'] ?? "No existe un mensaje configurado para el campo " . $label_pos['label'];
                array_push($messages_arr, $msg_to_arr);
              }
            }
          }

          if ($messages_arr) {
            throw new RekognitionException($messages_arr[0], $command);
          }
        }

        // Condiciones que debe tener.
        unset($config['container_positivos']['add']);
        $allowed_number = $config['allowed_number'];
        $cont = 0;
        if ($flag_neg && isset($config['container_positivos']) && !empty($config['container_positivos'])) {
          $message_err = "Tu foto no es una selfi. Por favor ubícate frente a tu celular, acerca más tu cara y tómate una foto.";
          foreach ($config['container_positivos'] as $label_pos) {
            $flag_exist = FALSE;
            foreach ($result['Labels'] as $label) {
              if ($label_pos['label'] == $label['Name']) {
                if ($label['Confidence'] >= $label_pos['percentage']) {
                  $flag_exist = TRUE;
                }
                else {
                  $percent_amazon = number_format($label['Confidence']);
                  array_push($percents_arr, $percent_amazon);
                  $msg_to_arr = $label_pos['error_description'] ?? "No existe un mensaje configurado para el campo " . $label_pos['label'];
                  $messages_arr[$percent_amazon] = $msg_to_arr;
                  $flag_exist = TRUE;
                  $flag_pos = FALSE;
                }
              }
            }
            if (!$flag_exist) {
              $flag_pos = FALSE;
              array_push($percents_arr, 0);
              $cont++;
              $messages_arr[] = $label_pos['error_description'] ?? "No existe un mensaje configurado para el campo " . $label_pos['label'];
              // break;
            }
          }

          if ($cont > $allowed_number) {
            // Se mantiene el error en etiquetas cuando supera el número de intentos permitidos.
            if ($percents_arr && $messages_arr) {
              $message_err = $messages_arr[min($percents_arr)];
            }
          }
          else {
            // Permite pasar
            $flag_pos = TRUE;
          }
        }

        if ($flag_neg && $flag_pos) {
          if (isset($config['multiuser']) && $config['multiuser']) {
            try {
              // Call DetectFaces.
              $result = $this->connect->DetectFaces([
                'Image' => [
                  'Bytes' => $image,
                ],
                'Attributes' => ['ALL'],
              ]);
              if (!empty($result['FaceDetails'])) {
                $cant = count($result['FaceDetails']);
                if ($cant == 1) {
                  $return = [
                    'send' => TRUE,
                  ];
                }
                else {
                  $msg_to_error = '<span class="err_contract" data-error="1"><b>Paso 1:</b></span> Hay más de una persona en la foto. Por favor tómala de nuevo y verifica que aparezcas solo tú.';
                  throw new RekognitionException($msg_to_error, $command);
                }
              }
              else {
                $msg_to_error = '<span class="err_contract" data-error="1"><b>Paso 1:</b></span> No hay ninguna persona en la foto. Por favor tómala de nuevo y verifica que aparezcas.';
                throw new RekognitionException($msg_to_error, $command);
              }

            }
            catch (RekognitionException $e) {
              $return = [
                'error' => $e->getMessage(),
              ];
            }
          }
          else {
            $return = [
              'send' => TRUE,
            ];
          }
        }
        else {
          $msg_to_error = '<span class="err_contract" data-error="1"><b>Paso 1: </b></span>' . $message_err;
          throw new RekognitionException($msg_to_error, $command);
        }

      }
      catch (RekognitionException $e) {
        $return = [
          'error' => $e->getMessage(),
        ];
      }

    }
    return $return;
  }

  /**
   * Validacion documento.
   *
   * Consulta al servicio de amazon, para detectar
   * que sea un documento y luego extraer el texto.
   *
   * @return mixed
   */
  public function document($image, $config, $prefix) {
    $this->connect();
    $command = new Command('Error');
    try {
      $result = $this->connect->detectLabels([
        'Image' => [
          'Bytes' => $image,
        ],
        'Attributes' => ['ALL'],
      ]);
      // Condiciones que no debe tener.
      unset($config[$prefix . '_negativo']['add']);
      if (isset($config[$prefix . '_negativo'])
      && !empty($config[$prefix . '_negativo'])) {
        foreach ($result['Labels'] as $label) {
          foreach ($config[$prefix . '_negativo'] as $label_neg) {
            if (isset($label_neg['label'])
            && $label['Name'] == $label_neg['label']) {
              $msg_to_error = $label_neg['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_neg['label'];
              throw new RekognitionException($msg_to_error, $command);
            }
          }
        }
      }

      // Porcentaje.
      unset($config[$prefix . '_positivo']['add']);
      if (isset($config[$prefix . '_positivo'])
      && !empty($config[$prefix . '_positivo'])) {
        foreach ($config[$prefix . '_positivo'] as $label_pos) {
          $flag_exist = FALSE;
          foreach ($result['Labels'] as $label) {
            if ($label['Name'] == $label_pos['label']) {
              if ($label['Confidence'] >= $label_pos['percentage']) {
                $flag_exist = TRUE;
              }
              else {
                $msg_to_error = $label_pos['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_pos['label'];
                throw new RekognitionException($msg_to_error, $command);
              }
            }
          }
          if (!$flag_exist) {
            $msg_to_error = $label_pos['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_pos['label'];
            throw new RekognitionException($msg_to_error, $command);
          }
        }
      }
      $text = $this->text($image);
      if (!isset($text['error'])) {
        // Expresiones regulares que debe tener.
        $text = str_replace('.', '', $text);
        $return = str_replace(' ', '', $text);
        unset($config[$prefix . '_regular_expresion']['add']);
        if (isset($config[$prefix . '_regular_expresion'])
        && !empty($config[$prefix . '_regular_expresion'])) {
          foreach ($config[$prefix . '_regular_expresion'] as $label_regex) {
            if (!preg_match($label_regex['label'], $return)) {
              $msg_to_error = $label_regex['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_regex['label'];
              throw new RekognitionException($msg_to_error, $command);
            }
          }
        }
      }
      else {
        $return = $text;
      }
    }
    catch (RekognitionException $e) {
      $return = [
        'error' => $e->getMessage(),
      ];
    }
    return $return;
  }

  /**
   * Detectar texto.
   */
  public function text($image, $nection = FALSE) {
    // Call DetectText.
    if ($nection) {
      $this->connect();
    }
    try {
      $command = new Command('Error');
      $result = $this->connect->detectText([
        'Image' => [
          'Bytes' => $image,
        ],
        'Attributes' => ['ALL'],
      ]);
      if (!empty($result['TextDetections'])) {
        $return = NULL;
        foreach ($result['TextDetections'] as $text) {
          $return .= $text['DetectedText'] . ' ';
        }
      }
      else {
        $msg_to_error = '<span class="err_contract" data-error="3"><b>Paso 3: </b></span>Tu documento no es una cédula. Por favor ubica tu cédula en una superficie plana y tómale una foto.';
        throw new RekognitionException($msg_to_error, $command);
      }
    }
    catch (RekognitionException $e) {
      $return = [
        'error' => $e->getMessage(),
      ];
    }
    return $return;
  }

}
