<?php

namespace Drupal\amazon_image_id_scan\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\image\Plugin\Field\FieldWidget\ImageWidget as BaseImageWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\FormValidationException;

/**
 * Plugin implementation of the 'image_widget' widget.
 *
 * @FieldWidget(
 *   id = "image_widget",
 *   label = @Translation("Mi Widget Imagen"),
 *   field_types = {
 *     "image"
 *   },
 * )
 */
class ImageScanWidget extends BaseImageWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'progress_indicator' => 'throbber',
      'preview_image_style' => 'thumbnail',
      'config_id_scan' => NULL,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::settingsForm($form, $form_state);
    $config = \Drupal::config('amazon_image_id_scan.settings');
    $config_rekognition = $config->get('rekognition_tab');
    $options = [];
    if (isset($config_rekognition['tab_config']) && is_array($config_rekognition['tab_config'])) {
      foreach ($config_rekognition['tab_config'] as $key => $value) {
        if (is_numeric($key)) {
          $options[$key] = $value['label'];
        }
      }
    }
    $element['preview_image_style'] = [
      '#title' => $this->t('Preview image style'),
      '#type' => 'select',
      '#options' => image_style_options(FALSE),
      '#empty_option' => '<' . $this->t('no preview') . '>',
      '#default_value' => $this->getSetting('preview_image_style'),
      '#description' => $this->t('The preview image will be shown while editing the content.'),
      '#weight' => 15,
    ];
    $element['config_id_scan'] = [
      '#title' => $this->t('Config id scan'),
      '#type' => 'select',
      '#options' => $options,
      '#empty_option' => '<' . $this->t('no config') . '>',
      '#default_value' => $this->getSetting('config_id_scan'),
      '#description' => $this->t('Image Scan Settings.'),
      '#weight' => 15,
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $field_settings = $this->getFieldSettings();

    // Add image validation.
    $element['#upload_validators']['FileIsImage'] = [];

    // Add upload dimensions validation.
    if ($field_settings['max_resolution'] || $field_settings['min_resolution']) {
      $element['#upload_validators']['FileImageDimensions'] = [
        'maxDimensions' => $field_settings['max_resolution'],
        'minDimensions' => $field_settings['min_resolution'],
      ];
    }

    $extensions = $field_settings['file_extensions'];
    $supported_extensions = $this->imageFactory->getSupportedExtensions();

    // If using custom extension validation, ensure that the extensions are
    // supported by the current image toolkit. Otherwise, validate against all
    // toolkit supported extensions.
    $extensions = !empty($extensions) ? array_intersect(explode(' ', $extensions), $supported_extensions) : $supported_extensions;
    $element['#upload_validators']['FileExtension']['extensions'] = implode(' ', $extensions);

    // Add mobile device image capture acceptance.
    $element['#accept'] = 'image/*';

    // Add properties needed by process() method.
    $element['#preview_image_style'] = $this->getSetting('preview_image_style');
    $element['#title_field'] = $field_settings['title_field'];
    $element['#title_field_required'] = $field_settings['title_field_required'];
    $element['#alt_field'] = $field_settings['alt_field'];
    $element['#alt_field_required'] = $field_settings['alt_field_required'];

    // Default image.
    $default_image = $field_settings['default_image'];
    if (empty($default_image['uuid'])) {
      $default_image = $this->fieldDefinition->getFieldStorageDefinition()->getSetting('default_image');
    }
    // Convert the stored UUID into a file ID.
    if (!empty($default_image['uuid']) && $entity = \Drupal::service('entity.repository')->loadEntityByUuid('file', $default_image['uuid'])) {
      $default_image['fid'] = $entity->id();
    }
    $element['#default_image'] = !empty($default_image['fid']) ? $default_image : [];
    $element['#element_validate'] = [[$this, 'validateScan']];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateScan(array &$element, FormStateInterface $form_state) {
    $name = $form_state->getTriggeringElement()['#name'];
    if (strpos($name, 'upload_button') !== FALSE) {
      $value = $form_state->getValue($element['#field_name']);
      foreach ($value as $fid) {
        if (isset($fid['fids'][0])) {
          $file = \Drupal\file\Entity\File::load($fid['fids'][0]);
          if ($file) {
            $url = $file->createFileUrl(FALSE);
            $image = file_get_contents($url);
            if ($image) {
              $id_config = $this->getSetting('config_id_scan');
              if ($id_config && is_numeric($id_config)) {
                $service = \Drupal::service('amazon_image_id_scan.rekognition');
                $config = \Drupal::config('amazon_image_id_scan.settings');
                $config_scan = $config->get($id_config);
                $response = $service->document($image, $config_scan, $id_config);
                if (isset($response['error'])) {
                  $form_state->setError($element, $response['error']);
                }
              }

            }
          }
        }
      }
    }
  }

}