<?php

namespace Drupal\amazon_image_id_scan\Form;

use Drupal\Core\Form\ConfigFormBase;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines a form that configures forms module settings.
 * @codeCoverageIgnore
 */
class ConfigurationForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'amazon_image_id_scan_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'amazon_image_id_scan.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, Request $request = NULL) {

    $config = $this->config('amazon_image_id_scan.settings');

    $form['#attached'] = [
      'library' => [
        'amazon_image_id_scan/amazon_image_id_scan.main',
      ],
    ];
    $form['#tree'] = TRUE;

    $form['bootstrap'] = [
      '#type' => 'vertical_tabs',
      '#prefix' => '<h2><small>' . $this->t('Amazon rekognition') . '</small></h2>',
      '#weight' => -10,
      '#default_tab' => 'rekognition_tab',
    ];

    $config_rekognition = $config->get('rekognition_tab');
    $form['rekognition_tab'] = [
      '#type' => 'details',
      '#title' => $this->t('Configuración credenciales Rekognition'),
      '#group' => 'bootstrap',
    ];

    $form['rekognition_tab']['key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('key'),
      '#default_value' => $config_rekognition['key'] ?? FALSE,
      '#required' => 1,
    ];

    $form['rekognition_tab']['secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret'),
      '#default_value' => $config_rekognition['secret'] ?? NULL,
      '#required' => 1,
    ];
    $form['rekognition_tab']['test'] = [
      '#type' => 'markup',
      '#markup' => 'Puede ver los diferentes las etiquetas, en el siguiente <a href="https://us-east-2.console.aws.amazon.com/rekognition/home?region=us-east-2#/label-detection" target="_blank">link</a>',
    ];
    $form['rekognition_tab']['tab_config'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'tab_config',
        'class' => 'container-tab-config',
      ],
    ];
    $tab_config = $config_rekognition['tab_config'] ?? FALSE;

    $this->multipleField($form, $form_state, $form['rekognition_tab']['tab_config'], $tab_config, 'tabConfig', 'tab_config');
    if ($tab_config && is_array($tab_config)) {
      foreach ($tab_config as $key => $value) {
        if (is_numeric($key)) {
          $configuration = $config->get($key);
          $form[$key] = [
            '#type' => 'details',
            '#title' => $value['label'],
            '#group' => 'bootstrap',
          ];
          $form[$key]['title_positivo'] = [
            '#type' => 'markup',
            '#markup' => "<h3>Etiquetas que debe tener {$value['label']}</h3>",
          ];
          $form[$key][$key . '_positivo'] = [
            '#type' => 'container',
            '#attributes' => [
              'id' => $key . '_positivo',
              'class' => [
                'container-group',
                'container-positivo',
              ],
            ],
          ];
          $positivo = $configuration[$key . '_positivo'] ?? FALSE;
          $this->multipleField($form, $form_state, $form[$key][$key . '_positivo'], $positivo, 'containerPositivos', $key . '_positivo');
          $form[$key]['title_negativo'] = [
            '#type' => 'markup',
            '#markup' => "<h3>Etiquetas que NO debe tener {$value['label']}</h3>",
          ];

          $form[$key][$key . '_negativo'] = [
            '#type' => 'container',
            '#attributes' => [
              'id' => $key . '_negativo',
              'class' => [
                'container-group',
                'container-negativo',
              ],
            ],
          ];
          $negativo = $configuration[$key . '_negativo'] ?? FALSE;
          $this->multipleField($form, $form_state, $form[$key][$key . '_negativo'], $negativo, 'containerNegativos', $key . '_negativo');
          $form[$key]['title_regular_expresion'] = [
            '#type' => 'markup',
            '#markup' => "<h3>Expresiones regulares para {$value['label']}</h3>",
          ];
          $form[$key][$key . '_regular_expresion'] = [
            '#type' => 'container',
            '#attributes' => [
              'id' => $key . '_regular_expresion',
              'class' => [
                'container-group',
                'container-negativo',
              ],
            ],
          ];
          $regular_expresion = $configuration[$key . '_regular_expresion'] ?? FALSE;
          $this->multipleField($form, $form_state, $form[$key][$key . '_regular_expresion'], $regular_expresion, 'containerRegularExpresion', $key . '_regular_expresion');
        }
      }
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $rekognition_tab = $form_state->getValue('rekognition_tab');
    $config = $this->config('amazon_image_id_scan.settings');
    $config->set('rekognition_tab', $rekognition_tab);
    if (isset($rekognition_tab['tab_config']) && is_array($rekognition_tab['tab_config'])) {
      foreach ($rekognition_tab['tab_config'] as $key => $value) {
        if (is_numeric($key)) {
          $config->set($key, $form_state->getValue($key));
        }
      }
    }
    $config->save();
  }

  /**
   * {@inheritdoc}
   */
  public function multipleField(array &$form, &$form_state, &$form_content, $default_value, $function, $name) {
    $delete_item = $form_state->get('delete_item_' . $name);
    $max = 0;
    if ($default_value) {
      $details = $default_value;
      $max = count($details) - 1;
      if ($delete_item && is_array($delete_item)) {
        foreach ($delete_item as $delete) {
          unset($details[$delete]);
        }
      }
      foreach ($details as $key => $value) {
        $this->$function($form, $form_content, $key, $name, $value);
      }
    }
    $add = $form_state->get('add_' . $name);
    if ($add) {
      for ($i = 1; $i <= $add; $i++) {
        $count = $max + $i;
        if (!$delete_item || !in_array($count, $delete_item)) {
          $this->$function($form, $form_content, $count, $name);
        }
      }
    }
    if (isset($form_content['add'])) {
      unset($form_content['add']);
    }
    $form_content['add'] = [
      '#type' => 'submit',
      '#value' => "Agregar item",
      '#submit' => ['::submitAddDetails'],
      '#ajax' => [
        'callback' => '::wrapperDetails',
        'wrapper' => $name,
      ],
      '#name' => $name,
      '#attributes' => [
        'class' => ['btn-add-item'],
      ],
    ];
  }

  /**
   * Contedor de etiquetas positivas.
   */
  public function containerPositivos(array &$form, &$form_content, $key, $function, $value = FALSE) {
    $form_content[$key]['label'] = [
      '#type' => 'textfield',
      '#title' => 'Etiqueta',
      '#default_value' => $value['label'] ?? NULL,
      '#size' => 15,
      '#maxlength' => 40,
    ];

    $form_content[$key]['percentage'] = [
      '#type' => 'number',
      '#title' => 'Porcentaje',
      '#default_value' => $value['percentage'] ?? NULL,
      '#max' => '100',
      '#size' => 10,
      '#attributes' => [
        'class' => ['field-percent'],
      ],
    ];
    $form_content[$key]['error_description'] = [
      '#type' => 'textfield',
      '#title' => 'Descripción del error',
      '#default_value' => $value['error_description'] ?? NULL,
      '#size' => 45,
      '#maxlength' => 250,
    ];
    $form_content[$key]['delete'] = [
      '#type' => 'submit',
      '#value' => "Eliminar item",
      '#submit' => ['::submitDeleteItem'],
      '#name' => $key . '-' . $function,
      '#ajax' => [
        'callback' => '::wrapperDetails',
        'wrapper' => $function,
      ],
      '#attributes' => [
        'class' => ['button-delete-item'],
      ],
    ];
    $form_content[$key]['separate'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['separador'],
      ],
    ];
  }

  /**
   * Label negativos.
   */
  public function containerNegativos(array &$form, &$form_content, $key, $function, $value = FALSE) {
    $form_content[$key]['label'] = [
      '#type' => 'textfield',
      '#title' => 'Etiqueta',
      '#default_value' => $value['label'] ?? NULL,
      '#size' => 15,
      '#maxlength' => 40,
    ];
    $form_content[$key]['error_description'] = [
      '#type' => 'textfield',
      '#title' => 'Descripción del error',
      '#default_value' => $value['error_description'] ?? NULL,
      '#size' => 45,
      '#maxlength' => 250,
      '#attributes' => [
        'class' => ['field-description-item'],
      ],
    ];
    $form_content[$key]['delete'] = [
      '#type' => 'submit',
      '#value' => "Eliminar item",
      '#submit' => ['::submitDeleteItem'],
      '#name' => $key . '-' . $function,
      '#ajax' => [
        'callback' => '::wrapperDetails',
        'wrapper' => $function,
      ],
      '#attributes' => [
        'class' => ['button-delete-item'],
      ],
    ];
    $form_content[$key]['separate'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['separador'],
      ],
    ];
  }

  /**
   * Tab configuracion.
   */
  public function tabConfig(array &$form, &$form_content, $key, $function, $value = FALSE) {
    $form_content[$key]['label'] = [
      '#type' => 'textfield',
      '#title' => 'Etiqueta',
      '#default_value' => $value['label'] ?? NULL,
      '#size' => 15,
      '#maxlength' => 40,
    ];
    $form_content[$key]['delete'] = [
      '#type' => 'submit',
      '#value' => "Eliminar item",
      '#submit' => ['::submitDeleteItem'],
      '#name' => $key . '-' . $function,
      '#ajax' => [
        'callback' => '::wrapperDetails',
        'wrapper' => $function,
      ],
      '#attributes' => [
        'class' => ['button-delete-item'],
      ],
    ];
    $form_content[$key]['separate'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['separador'],
      ],
    ];
  }

  /**
   * Contedor para expresiones regulares.
   */
  public function containerRegularExpresion(array &$form, &$form_content, $key, $function, $value = FALSE) {
    $form_content[$key]['label'] = [
      '#type' => 'textfield',
      '#title' => 'Expresion regular',
      '#default_value' => $value['label'] ?? FALSE,
      '#size' => 50,
      '#maxlength' => 250,
    ];

    $form_content[$key]['error_description'] = [
      '#type' => 'textfield',
      '#title' => 'Descripción del error',
      '#default_value' => $value['error_description'] ?? NULL,
      '#size' => 45,
      '#maxlength' => 250,
    ];
    $form_content[$key]['delete'] = [
      '#type' => 'submit',
      '#value' => "Eliminar item",
      '#submit' => ['::submitDeleteItem'],
      '#name' => $key . '-' . $function,
      '#ajax' => [
        'callback' => '::wrapperDetails',
        'wrapper' => $function,
      ],
      '#attributes' => [
        'class' => ['button-delete-item'],
      ],
    ];
    $form_content[$key]['separate'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['separador'],
      ],
    ];
  }

  /**
   * Eliminar item.
   */
  public function submitDeleteItem(array &$form, FormStateInterface $form_state) {
    $name = $form_state->getTriggeringElement()['#name'];
    $explode = explode('-', $name);
    $tid = $explode[0];
    $var = 'delete_item_' . $explode[1];
    $add = $form_state->get($var);
    if ($add) {
      $add[] = $tid;
      $form_state->set($var, $add);
    }
    else {
      $add[] = $tid;
      $form_state->set($var, $add);
    }
    $form_state->setRebuild();
  }

  /**
   * Wrapper dinamico.
   */
  public function wrapperDetails(array &$form, FormStateInterface $form_state) {
    $parents = $form_state->getTriggeringElement()['#parents'];
    return $form[$parents[0]][$parents[1]];
  }

  /**
   * Contados de numero de items.
   */
  public function submitAddDetails(array &$form, FormStateInterface $form_state) {
    $name = $form_state->getTriggeringElement()['#name'];
    $var = 'add_' . $name;
    $add = $form_state->get($var);
    if ($add) {
      $add += 1;
      $form_state->set($var, $add);
    }
    else {
      $form_state->set($var, 1);
    }
    $form_state->setRebuild();
  }

}
