<?php

namespace Drupal\amazon_image_id_scan\Services;

use Aws\Command;
use Aws\Rekognition\Exception\RekognitionException;
use Aws\Rekognition\RekognitionClient;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Clase para la conexion a servicios de amazon.
 */
class Rekognition {
  /**
   * Variable para traer configuraciones.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * Variable con la conexion a amazon.
   *
   * @var Object
   */
  protected $connect;

  /**
   * Constructor del formulario.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->config = $config_factory->get('amazon_image_id_scan.settings');
  }

  /**
   * Conexion con amazon.
   */
  public function connect() {
    $credentials = $this->config->get('rekognition_tab');
    $client_config['credentials'] = [
      'key' => $credentials['key'] ?? FALSE,
      'secret' => $credentials['secret'] ?? FALSE,
    ];
    $client_config['region'] = 'us-east-1';
    $client_config['version'] = 'latest';

    $this->connect = new RekognitionClient($client_config);
    return $this;
  }

  /**
   * Validacion documento.
   *
   * Consulta al servicio de amazon, para detectar
   * que sea un documento y luego extraer el texto.
   *
   * @return mixed
   */
  public function document($image, $config, $prefix) {
    $this->connect();
    $command = new Command('Error');
    try {
      $result = $this->connect->detectLabels([
        'Image' => [
          'Bytes' => $image,
        ],
        'Attributes' => ['ALL'],
      ]);
      // Condiciones que no debe tener.
      unset($config[$prefix . '_negativo']['add']);
      if (isset($config[$prefix . '_negativo'])
      && !empty($config[$prefix . '_negativo'])) {
        foreach ($result['Labels'] as $label) {
          foreach ($config[$prefix . '_negativo'] as $label_neg) {
            if (isset($label_neg['label'])
            && $label['Name'] == $label_neg['label']) {
              $msg_to_error = $label_neg['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_neg['label'];
              throw new RekognitionException($msg_to_error, $command);
            }
          }
        }
      }

      // Porcentaje.
      unset($config[$prefix . '_positivo']['add']);
      if (isset($config[$prefix . '_positivo'])
      && !empty($config[$prefix . '_positivo'])) {
        foreach ($config[$prefix . '_positivo'] as $label_pos) {
          $flag_exist = FALSE;
          foreach ($result['Labels'] as $label) {
            if ($label['Name'] == $label_pos['label']) {
              if ($label['Confidence'] >= $label_pos['percentage']) {
                $flag_exist = TRUE;
              }
              else {
                $msg_to_error = $label_pos['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_pos['label'];
                throw new RekognitionException($msg_to_error, $command);
              }
            }
          }
          if (!$flag_exist) {
            $msg_to_error = $label_pos['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_pos['label'];
            throw new RekognitionException($msg_to_error, $command);
          }
        }
      }
      $text = $this->text($image);
      if (!isset($text['error'])) {
        // Expresiones regulares que debe tener.
        $text = str_replace('.', '', $text);
        $return = str_replace(' ', '', $text);
        unset($config[$prefix . '_regular_expresion']['add']);
        if (isset($config[$prefix . '_regular_expresion'])
        && !empty($config[$prefix . '_regular_expresion'])) {
          foreach ($config[$prefix . '_regular_expresion'] as $label_regex) {
            if (!preg_match($label_regex['label'], $return)) {
              $msg_to_error = $label_regex['error_description'] ?? "No hay un mensaje configurado para el campo " . $label_regex['label'];
              throw new RekognitionException($msg_to_error, $command);
            }
          }
        }
      }
      else {
        $return = $text;
      }
    }
    catch (RekognitionException $e) {
      $return = [
        'error' => $e->getMessage(),
      ];
    }
    return $return;
  }

  /**
   * Detectar texto.
   */
  public function text($image, $nection = FALSE) {
    // Call DetectText.
    if ($nection) {
      $this->connect();
    }
    try {
      $command = new Command('Error');
      $result = $this->connect->detectText([
        'Image' => [
          'Bytes' => $image,
        ],
        'Attributes' => ['ALL'],
      ]);
      if (!empty($result['TextDetections'])) {
        $return = NULL;
        foreach ($result['TextDetections'] as $text) {
          $return .= $text['DetectedText'] . ' ';
        }
      }
      else {
        $msg_to_error = '<span class="err_contract" data-error="3"><b>Paso 3: </b></span>Tu documento no es una cédula. Por favor ubica tu cédula en una superficie plana y tómale una foto.';
        throw new RekognitionException($msg_to_error, $command);
      }
    }
    catch (RekognitionException $e) {
      $return = [
        'error' => $e->getMessage(),
      ];
    }
    return $return;
  }

}
