<?php

namespace Drupal\tigo_amazon\Unit\Services;

use Aws\Command;
use Aws\Rekognition\Exception\RekognitionException;
use Aws\Rekognition\RekognitionClient;
use Drupal\amazon_image_id_scan\Services\Rekognition;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\KernelTests\KernelTestBase;

/**
 * Pruebas unitarias para el servicio Rekognition.
 *
 * @group amazon_image_id_scan
 */
class RekognitionTest extends KernelTestBase {

  /**
   * Módulos a cargar para la prueba.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'amazon_image_id_scan',
  ];

  /**
   * El servicio bajo prueba.
   *
   * @var \Drupal\amazon_image_id_scan\Services\Rekognition
   */
  protected $rekognitionService;

  /**
   * Mock de ConfigFactoryInterface.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Mock de RekognitionClient.
   *
   * @var \Aws\Rekognition\RekognitionClient
   */
  protected $rekognitionClient;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Configurar los mocks.
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->rekognitionClient = $this->createMock(RekognitionClient::class);

    // Crear una instancia del servicio.
    $this->rekognitionService = new Rekognition($this->configFactory);

    // Inyectar el mock de RekognitionClient en el servicio.
    $reflection = new \ReflectionClass($this->rekognitionService);
    $property = $reflection->getProperty('connect');
    $property->setAccessible(TRUE);
    $property->setValue($this->rekognitionService, $this->rekognitionClient);
  }

  /**
   * Prueba el método connect.
   */
  public function testConnect() {
    // Configurar el mock de ConfigFactory.
    $config = $this->createMock(\Drupal\Core\Config\Config::class);
    $config->method('get')
      ->willReturn([
        'key' => 'test-key',
        'secret' => 'test-secret',
      ]);
    $this->configFactory->method('get')
      ->willReturn($config);

    // Ejecutar el método connect.
    $result = $this->rekognitionService->connect();

    // Verificar que el resultado es una instancia de Rekognition.
    $this->assertInstanceOf(Rekognition::class, $result);
  }

  /**
   * Prueba el método document con éxito.
   */
  public function testDocumentSuccess() {
    // Configurar el mock de ConfigFactory.
    $config = $this->createMock(\Drupal\Core\Config\Config::class);
    $config->method('get')
      ->willReturn([
        'key' => 'test-key',
        'secret' => 'test-secret',
      ]);
    $this->configFactory->method('get')
      ->willReturn($config);

    // Configurar el mock de RekognitionClient.
    $this->rekognitionClient->method('detectLabels')
      ->willReturn([
        'Labels' => [
          ['Name' => 'Document', 'Confidence' => 95],
        ],
      ]);
    $this->rekognitionClient->method('detectText')
      ->willReturn([
        'TextDetections' => [
          ['DetectedText' => '123456789'],
        ],
      ]);

    // Ejecutar el método document.
    $result = $this->rekognitionService->document('test-image', [
      'prefix_negativo' => [],
      'prefix_positivo' => [
        ['label' => 'Document', 'percentage' => 90],
      ],
      'prefix_regular_expresion' => [
        ['label' => '/^\d+$/', 'error_description' => 'El texto no es válido'],
      ],
    ], 'prefix');

    // Verificar que el resultado es el texto esperado.
    $this->assertEquals('123456789', $result);
  }

  /**
   * Prueba el método document con error.
   */
  public function testDocumentError() {
    // Configurar el mock de ConfigFactory.
    $config = $this->createMock(\Drupal\Core\Config\Config::class);
    $config->method('get')
      ->willReturn([
        'key' => 'test-key',
        'secret' => 'test-secret',
      ]);
    $this->configFactory->method('get')
      ->willReturn($config);

    // Configurar el mock de RekognitionClient para lanzar una excepción.
    $this->rekognitionClient->method('detectLabels')
      ->willThrowException(new RekognitionException('Error en la detección de etiquetas', new Command('Error')));

    // Ejecutar el método document.
    $result = $this->rekognitionService->document('test-image', [
      'prefix_negativo' => [],
      'prefix_positivo' => [
        ['label' => 'Document', 'percentage' => 90],
      ],
      'prefix_regular_expresion' => [
        ['label' => '/^\d+$/', 'error_description' => 'El texto no es válido'],
      ],
    ], 'prefix');

    // Verificar que el resultado contiene un error.
    $this->assertArrayHasKey('error', $result);
  }

  /**
   * Prueba el método text con éxito.
   */
  public function testTextSuccess() {
    // Configurar el mock de ConfigFactory.
    $config = $this->createMock(\Drupal\Core\Config\Config::class);
    $config->method('get')
      ->willReturn([
        'key' => 'test-key',
        'secret' => 'test-secret',
      ]);
    $this->configFactory->method('get')
      ->willReturn($config);

    // Configurar el mock de RekognitionClient.
    $this->rekognitionClient->method('detectText')
      ->willReturn([
        'TextDetections' => [
          ['DetectedText' => '123456789'],
        ],
      ]);

    // Ejecutar el método text.
    $result = $this->rekognitionService->text('test-image');

    // Verificar que el resultado es el texto esperado.
    $this->assertEquals('123456789 ', $result);
  }

  /**
   * Prueba el método text con error.
   */
  public function testTextError() {
    // Configurar el mock de ConfigFactory.
    $config = $this->createMock(\Drupal\Core\Config\Config::class);
    $config->method('get')
      ->willReturn([
        'key' => 'test-key',
        'secret' => 'test-secret',
      ]);
    $this->configFactory->method('get')
      ->willReturn($config);

    // Configurar el mock de RekognitionClient para lanzar una excepción.
    $this->rekognitionClient->method('detectText')
      ->willThrowException(new RekognitionException('Error en la detección de texto', new Command('Error')));

    // Ejecutar el método text.
    $result = $this->rekognitionService->text('test-image');

    // Verificar que el resultado contiene un error.
    $this->assertArrayHasKey('error', $result);
  }

}