import { Plugin } from "ckeditor5/src/core";
import { toWidget, Widget } from "ckeditor5/src/widget";

// cSpell:ignore insertasin insertasincommand

/** *
 * This file has the logic for defining the asinLink model, and for how it is
 * converted to standard DOM markup.
 */
export default class InsertAsinEditing extends Plugin {
  static get requires() {
    return [Widget];
  }

  init() {
    this._defineSchema();
    this._defineConverters();
  }

  /*
   * This registers the custom link structure that will be seen by CKEditor 5 as
   * asinLink.
   * The logic in _defineConverters() will determine how this is converted to
   * markup.
   */
  _defineSchema() {
    // Schemas are registered via the central `editor` object.
    const { schema } = this.editor.model;

    schema.register("asinLink", {
      // Behaves like a self-contained object (e.g. an image).
      isObject: true,
      isLimit: true,
      isSelectable: true,
      isInline: true,

      // Allow in places where other blocks are allowed (e.g. directly in the root).
      allowWhere: "$text",
      allowContentOf: "$block",
      allowAttributes: ["productId"],
    });
  }

  /**
   * Converters determine how CKEditor 5 models are converted into markup and
   * vice-versa.
   */
  _defineConverters() {
    // Converters are registered via the central editor object.
    const { conversion } = this.editor;

    // Upcast Converters: determine how existing HTML is interpreted by the
    // editor. These trigger when an editor instance loads.
    conversion.for("upcast").elementToElement({
      view: {
        name: "a",
        attributes: {
          href: true,
          "data-amazon-onsite-product": true,
        },
      },
      model: {
        key: "productId",
        value: (viewElement) =>
          viewElement.getAttribute("data-amazon-onsite-product-id"),
      },
    });

    // Data Downcast Converters: converts stored model data into HTML.
    // These trigger when content is saved.
    conversion
      .for("downcast")
      .attributeToAttribute({
        model: "url",
        view: "href",
      })
      .elementToElement({
        model: "asinLink",
        view: (modelElement, { writer: viewWriter }) => {
          const productId = modelElement.getAttribute("productId");
          const linkElement = viewWriter.createContainerElement(
            "a",
            {
              href: `https://www.amazon.de/dp/${productId}`,
              "data-amazon-onsite-product": true,
              "data-amazon-onsite-product-id": productId,
            },
            { priority: 5 }
          );
          const linkText = viewWriter.createText(`ASIN:${productId}`);

          viewWriter.insert(
            viewWriter.createPositionAt(linkElement, 0),
            linkText
          );

          return toWidget(linkElement, viewWriter, {
            label: "Amazon Product Widget",
          });
        },
      });
  }
}
