/**
 * @file registers the insertAsin toolbar button and binds functionality to it.
 */

import { Plugin } from "ckeditor5/src/core";
import {
  ButtonView,
  ContextualBalloon,
  clickOutsideHandler,
} from "ckeditor5/src/ui";
import icon from "../../../../icons/asinLink.svg";
import FormView from "./insertasinuiformview";

export default class InsertAsinUi extends Plugin {
  init() {
    const { editor } = this;

    this._balloon = this.editor.plugins.get(ContextualBalloon);
    this.formView = this._createFormView();

    // This will register the simpleBox toolbar button.
    editor.ui.componentFactory.add("insertAsin", (locale) => {
      const buttonView = new ButtonView(locale);
      buttonView.set({
        label: editor.t("Insert amazon product card"),
        icon,
        tooltip: true,
      });

      // Show the UI on button click.
      this.listenTo(buttonView, "execute", () => {
        this._showUI();
      });

      return buttonView;
    });
  }

  _createFormView() {
    const { editor } = this;
    const formView = new FormView(editor.locale);

    // Execute the command after clicking the "Save" button.
    this.listenTo(formView, "submit", () => {
      // Grab values from the productId input field.
      const productId = formView.productId.fieldView.element.value;
      editor.model.change((writer) => {
        const link = writer.createElement("asinLink", {
          productId,
        });

        editor.model.insertContent(link, editor.model.document.selection);
      });

      // Hide the form view after submit.
      this._hideUI();
    });

    // Hide the form view after clicking the "Cancel" button.
    this.listenTo(formView, "cancel", () => {
      this._hideUI();
    });

    // Hide the form view when clicking outside the balloon.
    clickOutsideHandler({
      emitter: formView,
      activator: () => this._balloon.visibleView === formView,
      contextElements: [this._balloon.view.element],
      callback: () => this._hideUI(),
    });

    return formView;
  }

  _showUI() {
    const { model } = this.editor;
    const selectedContent = model
      .getSelectedContent(model.document.selection)
      .getChild(0);
    if (selectedContent && selectedContent.hasAttribute("productId")) {
      this.formView.productId.fieldView.value =
        selectedContent.getAttribute("productId");
    }

    this._balloon.add({
      view: this.formView,
      position: this._getBalloonPositionData(),
    });

    this.formView.focus();
  }

  _hideUI() {
    // Clear the input field values and reset the form.
    // this.formView.productId.fieldView.value = '';
    this.formView.element.reset();

    this._balloon.remove(this.formView);

    // Focus the editing view after inserting the abbreviation so the user can start typing the content
    // right away and keep the editor focused.
    this.editor.editing.view.focus();
  }

  _getBalloonPositionData() {
    const { view } = this.editor.editing;
    const viewDocument = view.document;
    let target = null;

    // Set a target position by converting view selection range to DOM
    target = () =>
      view.domConverter.viewRangeToDom(viewDocument.selection.getFirstRange());

    return {
      target,
    };
  }
}
