<?php

/**
 * @file
 * Amazon Admin pages.
 */
/**
 * Basic Amazon settings form.
 */
function amazon_pa_settings_form($form, &$form_state) {
  $cache = amazon_pa_data_cache();
  $options = array();

  foreach ($cache['locales'] as $locale => $data) {
    $options[$locale] = $data['name'];
  }

  $form['amazon_default_locale'] = array(
    '#type' => 'select',
    '#title' => t('Default Amazon locale'),
    '#default_value' => variable_get('amazon_default_locale', 'US'),
    '#options' => $options,
    '#description' => t('Amazon.com uses separate product databases and Ecommerce features in different locales; pricing and availability information, as well as product categorization, differs from one locale to the next. Be sure to select the default locale your site will be running in.'),
  );

  $form['associate_setting'] = array(
    '#type' => 'fieldset',
    '#title' => t('Amazon referral settings'),
    '#description' => t('Enter your associate ID for each locale you\'d like to use on your site. Locales left blank will not be available.<br><strong>Note:</strong> By default the Drupal Association\'s ID will populate the field if it\'s available unless you have already specified your own or cleared the value.'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );

  foreach ($cache['locales'] as $locale => $data) {
    $form['associate_setting']['amazon_locale_'. $locale .'_associate_id'] = array(
      '#type' => 'textfield',
      '#title' => t(':locale_name associate ID', array(':locale_name' => $data['name'])),
      '#description' => t('Enter your :locale_name associate ID to receive referral bonuses when shoppers purchase Amazon products via your site.', array(':locale_name' => $data['name'])),
      '#default_value' => variable_get('amazon_locale_'. $locale .'_associate_id', $data['da_associate_id']),
    );
  }

  $form['amazon_aws_access_key'] = array(
    '#type' => 'textfield',
    '#title' => t('Amazon AWS Access Key ID'),
    '#description' => t('You must sign up for an Amazon AWS account to use the Product Advertising Service. See !more_info for information and a registration form.', array('!more_info' => l(t('the AWS home page'), 'https://aws-portal.amazon.com/gp/aws/developer/account/index.html?ie=UTF8&action=access-key', array('html' => TRUE)))),
    '#default_value' => variable_get('amazon_aws_access_key', ''),
    '#required' => TRUE,
  );

  $form['amazon_aws_secret_access_key'] = array(
    '#type' => 'textfield',
    '#title' => t('Amazon AWS Secret Access Key'),
    '#description' => t('You must sign up for an Amazon AWS account to use the Product Advertising Service. See !more_info for information and a registration form.', array('!more_info' => l(t('the AWS home page'), 'https://aws-portal.amazon.com/gp/aws/developer/account/index.html?ie=UTF8&action=access-key', array('html' => TRUE)))),
    '#default_value' => variable_get('amazon_aws_secret_access_key', ""),
    '#required' => TRUE,
  );

  return system_settings_form($form);
}

/**
 * Validate the keys and trim whitespace.
 */
function amazon_pa_settings_form_validate($form, &$form_state) {
  $form_state['values']['amazon_aws_access_key'] = trim($form_state['values']['amazon_aws_access_key']);
  $form_state['values']['amazon_aws_secret_access_key'] = trim($form_state['values']['amazon_aws_secret_access_key']);

  // Load locale data
  require_once(drupal_get_path('module', 'amazon_pa') .'/includes/amazon_pa.locales.inc');
  $locales = _amazon_pa_load_locales();

  $expire = FALSE;
  // Set up default values for each locale to the Drupal Association ID for that locale (if available)
  foreach ($locales as $locale => $data) {
    $assoc_id = variable_get('amazon_locale_'. $locale .'_associate_id', '');
    if ($assoc_id != $form_state['values']['amazon_locale_'. $locale .'_associate_id']) {
      $expire = TRUE;
    }
  }

  if ($expire) {
    amazon_pa_expire_items();
  }
}

/**
 * Database cleanup operations
 *
 * @param $form
 * @param $form_state
 *
 * @return mixed
 */

function amazon_pa_database_form($form, &$form_state) {

  $form['database_info'] = array(
    '#markup' => '<p><b>Invalid ASINS should be removed to avoid throtteling issues. This pages removes them from the "items_table". Remember to change the nodes using this ASINS in a field or token too, or they will be regenerated!</b></p>'
  );

  //delete a single asin
  $form['asin_delete'] = array(
    '#type' => 'textfield',
    '#title' => t('Delete a single Amazon ASIN from database'),
    '#description' => t('The ASIN of a product listed on Amazon.'),
    '#required' => FALSE,
  );
  $form['asin_delete_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Delete ASIN'),
    '#submit' => array('amazon_pa_database_form_submit'),
    '#validate' => array('amazon_pa_database_form_validate'),
  );

  //delete all invalid asins from amazon_items table
  $form['invalid_asins'] = array (
    '#type' => 'fieldset',
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );
  $form['invalid_asins']['delete_invalid_asins']= array(
    '#type' => 'submit',
    '#value' => t('Delete ALL invalid ASINs from database'),
    '#submit' => array('amazon_pa_delete_invalid_all_asins_submit'),
    '#validate' => array('amazon_pa_delete_invalid_all_asins_validate'),
  );

  return $form;

}


/**
 * Form to determine how long things get cached.
 */
function amazon_pa_storage_settings_form($form, &$form_state) {

  $period = drupal_map_assoc(array(3600, 7200, 14400, 21600, 43200, 86400), 'format_interval');
  $form['details']['amazon_refresh_schedule'] = array(
    '#type' => 'select',
    '#title' => t('Amazon refresh schedule'),
    '#description' => t('Cached information on Amazon items must be refreshed regularly to keep pricing and stock information up to date. Cron must be enabled for this function to work properly.'),
    '#default_value' => variable_get('amazon_refresh_schedule', 86400),
    '#options' => $period
  );
  $form['details']['amazon_refresh_cron_limit'] = array(
    '#type' => 'textfield',
    '#title' => t('Limit'),
    '#description' => t('This will be the number of ASINS! processed each time cron runs.'),
    '#default_value' => variable_get('amazon_refresh_cron_limit', 10),
  );
  $form['debug'] = array (
    '#type' => 'fieldset',
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );
  $form['debug']['amazon_request_delay'] = array(
    '#type' => 'textfield',
    '#title' => t('DEBUG: Delay between requests'),
    '#description' => t('<b>This also slows down the page load by this amount! Only use for debug purposes</b>. This will fore a delay between two requests. Useful when you are NEW and get throtteled errors "too many requests", see logfiles. '),
    '#default_value' => variable_get('amazon_request_delay', 0),
  );
  $form['debug']['amazon_request_amount'] = array(
    '#type' => 'textfield',
    '#title' => t('DEBUG: Requests sent'),
    '#description' => t('Counting requests for debug purpose.'),
    '#default_value' => variable_get('amazon_request_amount', 0),
  );
  $form['debug']['amazon_request_amount_checkbox'] = array(
    '#type' => 'checkbox',
    '#title' => t('Activate request counter'),
    '#description' => t('Counting requests for debug purpose.'),
    '#default_value' => variable_get('amazon_request_amount_checkbox', 0),
  );
  $form['debug']['amazon_only_nodes'] = array(
    '#type' => 'checkbox',
    '#title' => t('Trigger API request only on a node. Meaning node/x'),
    '#description' => t('Useful for rebuilding the whole amazon_items table with a webcrawler like linkchecker. <b>DEACTIVATE AFTER!</b>'),
    '#default_value' => variable_get('amazon_only_nodes', 0),
  );
  $form['details']['amazon_core_data'] = array(
    '#type' => 'checkboxes',
    '#title' => t('Store extended product data'),
    '#default_value' => variable_get('amazon_core_data', array('creators', 'images')),
    '#options' => array(
      'creators' => t('Book authors, film actors, etc.'),
      'images' => t('Product image links'),
    ),
  );

  return system_settings_form($form);
}

/**
 * Form for testing a live-query with a single ASIN.
 *
 */
function amazon_pa_test_form($form, &$form_state) {
  $form = array();

  $cache = amazon_pa_data_cache();
  $options = array();
  foreach ($cache['locales'] as $locale => $data) {
    if (variable_get('amazon_locale_'. $locale .'_associate_id', '')) {
      $options[$locale] = $data['name'];
    }
  }

  $form['asin'] = array(
    '#type' => 'textfield',
    '#title' => t('Amazon Product ID'),
    '#description' => t('The ASIN of a product listed on Amazon.'),
    '#required' => TRUE,
  );
  $form['locale'] = array(
    '#type' => 'select',
    '#title' => t('Locale'),
    '#options' => $options,
    '#default_value' => variable_get('amazon_default_locale', 'US'),
  );
  $form['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Look Up Product'),
  );

  if (isset($form_state['amazon_item'])) {
    $form['item'] = array(
      '#type' => 'fieldset',
      '#title' => t('Result'),
      '#collapsible' => FALSE,
    );
    $form['item']['display'] = array(
      '#markup' => theme('amazon_item', array('item' => $form_state['amazon_item'], 'style' => 'test')),
      '#weight' => 9,
    );
    $form['item']['details'] = array(
      '#type' => 'fieldset',
      '#title' => t('Details'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
      '#weight' => 10,
    );
    $form['item']['details']['data'] = array(
      '#markup' => '<pre><small>' . print_r($form_state['amazon_item'], TRUE) . '</small></pre>',
    );
  }
  return $form;
}

/**
 * Form validate
 *
 * @param $form
 * @param $form_state
 */

function amazon_pa_test_form_validate($form, &$form_state) {

  // Normalize input and look up
  //this is risky and also can cause throtteling. remove
  //$asin = amazon_convert_to_asin($form_state['values']['asin']);
  $asin = $form_state['values']['asin'];

  $items = amazon_pa_item_lookup_from_web(array($asin), $form_state['values']['locale']);
  if (!empty($items) && is_array($items)) {
    $form_state['amazon_item'] = array_pop($items);
  }
  else {
    form_set_error('asin', t("%input is no valid ASIN. Please check the !link for messages.", array('%input' => $form_state['values']['asin'], '!link' => l(t("error log"), 'admin/reports/dblog'))));
  }
}

/**
 * Form SUbmit handler
 *
 * @param $form
 * @param $form_state
 */

function amazon_pa_test_form_submit($form, &$form_state) {
  $item = $form_state['amazon_item'];
  amazon_pa_item_delete($item['asin'], $item['locale']);
  amazon_pa_item_insert($item);
  $form_state['rebuild'] = TRUE;
}

/**
 * Form validate
 *
 * @param $form
 * @param $form_state
 */

function amazon_pa_database_form_validate($form, &$form_state) {

  $asin = $form_state['values']['asin_delete'];

  $items = amazon_pa_item_lookup_from_db(array($asin));
  if (!empty($items) && is_array($items)) {
    $form_state['amazon_item'] = array_pop($items);
  }
  else {
    form_set_error('asin', t("%input not found in database ASIN. Please check the !link for messages.", array('%input' => $form_state['values']['asin'], 'admin/reports/dblog')));
  }
}


/**
 * Form Submit handler
 *
 * @param $form
 * @param $form_state
 */
function amazon_pa_database_form_submit($form, &$form_state) {
  $item = $form_state['amazon_item'];
  drupal_set_message(t('Successfully deleted data for @asin for all locales.', array('@asin' => $item['asin'])), 'status');

  $cache = amazon_pa_data_cache();
  foreach ($cache['locales'] as $locale => $locale_data) {
    amazon_pa_item_delete($item['asin'], $locale);
  }

  $form_state['rebuild'] = TRUE;
}


function amazon_pa_delete_invalid_all_asins_validate($form, &$form_state) {
  $invalid_asins = amazon_pa_get_invalid_asins();
  if(empty($invalid_asins)) {
    form_set_error('asin', t("No invlaid ASINs to delete."));
  }
}


function amazon_pa_delete_invalid_all_asins_submit($form, &$form_state) {
  $stop=1;
  $deleted_asins = array();
  $invalid_asins = amazon_pa_get_invalid_asins();

  $cache = amazon_pa_data_cache();
  foreach ($invalid_asins as $key => $asin_data) {
    foreach ($cache['locales'] as $locale => $locale_data) {
      amazon_pa_item_delete($asin_data->asin, $locale);
      $deleted_asins[] = $asin_data->asin;
    }
  }

  $deleted_asins = implode(', ', array_unique($deleted_asins));

  drupal_set_message(t('Successfully deleted the following invalid ASINs @asins for all locales.', array('@asins' => $deleted_asins)), 'status');

}
