<?php

namespace Drupal\amazon_pa_filter\Plugin\Filter;

use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\FilterProcessResult;
use Drupal\Core\Form\FormStateInterface;

/**
 * @Filter(
 *   id = "amazon",
 *   title = @Translation("Amazon filter"),
 *   description = @Translation("Provides access to many types of Amazon data. Simplest usage: [amazon:ASIN:inline], for example [amazon:0596515804:inline]."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_MARKUP_LANGUAGE,
 * )
 */

class AmazonFilter extends FilterBase {

  /**
   * process the filter
   *
   * @param string $text
   * @param string $langcode
   *
   * @return \Drupal\filter\FilterProcessResult
   */

  public function process($text, $langcode) {

    //make this global and use it as a kind of cache for one run. On the first run we generate all ASIN data and reuse it for every other run!
    //that is because the filter will run for every ASIN. Meaning 5 tokens found = 5 runs. Would reusult in API hammering...very ineffective
    //we then check if the asin data is already present
    global $amz_items;
    global $asin_list;

    $pattern = "@\[amazon(?: |:|%20)+(.*?)(?:(?: |:|%20)(.*?))?\]@";
    $matches = array();

    if (preg_match_all($pattern, $text, $matches)) {
      $search = $matches[0];
      $replace = array();

      //MULTIPLE ASINS in one request. Solo triggers throtteling....VERY fast if multiple tokens on one page
      //if someone overuses tokens on a page..then we need to raise the random sleep
      //that may be a problem in php timeout and is not really great
      //no other way if you are new and only get 1 request per second.
      $asin_count = 0;

      //skip this if we have it cached already!
      if(empty($amz_items) ) {
        $amz_items = array();
        $asin_list = array();

        //all asisn to an array, max 10, then request to amazonapi
        //if more than 10, request then merge, then loop again until finish
        foreach ($matches[0] as $key => $value) {
          if ($asin_count <= 9) {
            $asin_list[$key] = $matches[1][$key];
            $asin_count++;
          }
          else {
            $amz_items = amazon_pa_item_lookup($asin_list) + $amz_items;
            $asin_count = 0;
          }
        }
        $amz_items = amazon_pa_item_lookup($asin_list) + $amz_items;
      }
      else {
        //we have all cached already, no need to get all data again for every used token.
      }

      $processed_asisn = array();

      //put back in the data for duplicate asins
      $amz_items_results = $amz_items;
      $new_items = array();

      foreach ($asin_list as $key => $asin) {
        $new_items[] = $amz_items_results[$asin];
      }

      //get all "actions" (detail, thumbnail, fill and so on) we later run for the replacement of tokens
      foreach ($new_items as $action_key => $item) {
        $asin = $item['asin'];
        $processed_asisn[] = $asin;
        $action = $matches[2][$action_key];
        if(!isset($matches[2][$action_key])) {
          $stop=1;
        }

        //renderer result. somehow all except amazon_detail produce an array with [0] key.
        //there must be a better way to do this cleaner here.

        switch ($action) {
          case "":
          case 'inline':
            $image_ = [
              '#theme' => 'amazon_pa_item',
              '#item' => $item,
            ];
            $render_ = \Drupal::service('renderer')->render($image_);
            //here again is no subarray like in other cases so no [0]
            $replace[] = trim($render_->__toString());
            break;

          // Full is a synonym of 'details'.
          case 'full':
          case 'details':
            $image_ = [
              '#theme' => 'amazon_details',
              '#item' => $item,
            ];
            $render_ = \Drupal::service('renderer')->render($image_);
            $replace[] = trim($render_[0]->__toString());
            break;

          // Handle themeable cases, like thumbnail.
          case 'thumbnail':
            $image_ = [
              '#theme' => 'amazon_item_thumbnail_medium',
              '#item' => $item,
            ];
            $render_ = \Drupal::service('renderer')->render($image_);
            $replace[] = trim($render_[0]->__toString());
            break;

          default:
            // Allow to use anything found in the item array. Like title, asin, detailpageurl $action contains the variable name
            $image_ = [
              '#theme' => 'amazon_detail',
              '#item' => $item,
              '#detail' => $action
            ];
            $render_ = \Drupal::service('renderer')->render($image_);
            //here again is no subarray like in other cases so no [0]
            $replace[] = trim($render_->__toString());
            break;
        }
      }

      //asins where we have no results or invalid must be marked
      // error case
      foreach ($asin_list as $asin) {
        if(!in_array($asin, $processed_asisn)) {
          $replace[] = "<!-- The Amazon product '$asin' could not be found.-->";
        }
      }

      $text = str_replace($search, $replace, $text);

    }

    return new FilterProcessResult($text);
  }

  public function settingsForm(array $form, FormStateInterface $form_state) {
    //not needed
    return $form;
  }




}
