<?php

namespace Drupal\amazon_ses_log\EventSubscriber;

use Drupal\amazon_ses\Event\MailSentEvent;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribe to mail sent events.
 */
class MailSentEventSubscriber implements EventSubscriberInterface {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a MailSentEventSubscriber object.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(Connection $connection, TimeInterface $time) {
    $this->database = $connection;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      MailSentEvent::SENT => 'logMessage',
    ];
  }

  /**
   * Add a sent message to the log.
   *
   * @param \Drupal\amazon_ses\Event\MailSentEvent $event
   *   The message event.
   */
  public function logMessage(MailSentEvent $event) {
    $email = $event->getEmail();

    $from = array_map(function ($address) {
      return $address->getAddress();
    }, $email->getFrom());

    $to = array_map(function ($address) {
      return $address->getAddress();
    }, $email->getTo());

    $body = $email->getTextBody();
    if (!$body) {
      $body = $email->getHtmlBody();
    }

    $this->database->insert('amazon_ses_log')
      ->fields([
        'message_id' => $event->getMessageId(),
        'from' => implode(', ', $from),
        'to' => implode(', ', $to),
        'subject' => $email->getSubject(),
        'body' => $body,
        'raw_message' => $email->toString(),
        'sent' => $this->time->getRequestTime(),
      ])
      ->execute();
  }

}
