<?php

namespace Drupal\amazon_ses\Plugin\Mail;

use Drupal\amazon_ses\AmazonSesHandlerInterface;
use Drupal\amazon_ses\MessageBuilderInterface;
use Drupal\Core\Mail\MailInterface;
use Drupal\Core\Mail\Plugin\Mail\PhpMail;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Amazon SES mail system plugin.
 *
 * @Mail(
 *   id = "amazon_ses_mail",
 *   label = @Translation("Amazon SES mailer"),
 *   description = @Translation("Sends an email using Amazon SES.")
 * )
 */
class AmazonSes extends PhpMail implements MailInterface, ContainerFactoryPluginInterface {

  public function __construct(
    protected $configFactory,
    protected AmazonSesHandlerInterface $handler,
    protected MessageBuilderInterface $messageBuilder,
    protected QueueFactory $queueFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $container->get('config.factory'),
      $container->get('amazon_ses.handler'),
      $container->get('amazon_ses.message_builder'),
      $container->get('queue'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function mail(array $message) {
    $config = $this->configFactory->get('amazon_ses.settings');

    if ($config->get('override_from') || !isset($message['from'])) {
      $name = $config->get('from_name');
      $address = $config->get('from_address');

      $message['from'] = "$name <$address>";
    }

    $email = $this->messageBuilder->buildMessage($message);
    $queue = $this->queueFactory->get('amazon_ses_mail_queue');
    $queue_enabled = $this->configFactory
      ->get('amazon_ses.settings')
      ->get('queue');

    if ($queue_enabled) {
      $result = $queue->createItem($email);

      return (bool) $result;
    }
    else {
      $message_id = $this->handler->send($email);

      return (bool) $message_id;
    }
  }

}
