<?php

namespace Drupal\amp_optimizer\EventSubscriber;

use AmpProject\Optimizer\ErrorCollection;
use AmpProject\Optimizer\TransformationEngine;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

class OptimizerSubscriber implements EventSubscriberInterface {

  /**
   * Response transformation engine.
   *
   * @var \AmpProject\Optimizer\TransformationEngine
   */
  protected $transformationEngine;

  /**
   * Configs.
   *
   * @var array|mixed
   */
  protected $config;

  /**
   * Logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Error collection.
   *
   * @var \AmpProject\Optimizer\ErrorCollection
   */
  protected $errorCollection;

  public function __construct(TransformationEngine $transformation_engine, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_channel_factory) {
    $this->transformationEngine = $transformation_engine;
    $this->config = $config_factory->get('amp_optimizer.settings')->get() ?? [];
    $this->logger = $logger_channel_factory->get('amp_optimizer');
    $this->errorCollection = new ErrorCollection();
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::RESPONSE => [
        ['optimizer'],
      ],
    ];
  }

  public function optimizer(ResponseEvent $event): void {
    if (
      !array_key_exists('transform_enabled', $this->config) ||
      FALSE === $this->config['transform_enabled'] ||
      !$event->isMasterRequest() ||
      !$this->isAmpHtml($event->getResponse(), $event->getRequest())
    ) {
      return;
    }

    $optimized_html = $this->transformationEngine->optimizeHtml(
      $event->getResponse()->getContent(),
      $this->errorCollection
    );

    $this->handleErrors();

    $event->getResponse()->setContent($optimized_html);
  }

  /**
   * @param Response $response
   * @param Request $request
   *
   * @return bool
   */
  private function isAmpHtml(Response $response, Request $request): bool {
    $pathInfo = pathInfo($request->getUri());
    if (isset($pathInfo['extension']) && $pathInfo['extension'] !== 'html') {
      return FALSE;
    }

    $contentType = $response->headers->get('Content-type');
    if (strpos($contentType, 'text/html') === FALSE) {
      return FALSE;
    }

    $content = $response->getContent();
    $dom = (new \DOMDocument())->loadHTML($content);

    if (!$dom->validate()) {
      $this->logger->error('Content can not be parsed by HtmlDomParser');
      return FALSE;
    }

    $htmlElement = $dom->getElementsByTagName('html')->item(0);
    if (NULL === $htmlElement || !$htmlElement->hasAttributes()) {
      return FALSE;
    }

    $htmlElementAttrs = $htmlElement->attributes;

    return $htmlElementAttrs->getNamedItem('⚡') !== NULL ||
      $htmlElementAttrs->getNamedItem('amp') !== NULL;
  }


  private function handleErrors(): void {
    if ($this->errorCollection->count() > 0) {
      foreach ($this->errorCollection as $error) {
        $this->logger->error(sprintf(
          "AMP-Optimizer Error code: %s\nError Message: %s\n",
          $error->getCode(),
          $error->getMessage()
        ));
      }
    }
  }

}
