<?php
/**
 * AMQPInstance Class
 *
 * This is simply a wrapper around a connection, channel and exchange
 * to make interacting with the AMQP server easier. 
 */
class AMQPInstance {

  public static $messages = array();
  public static $connection = NULL;
  public static $channel = NULL;
  public static $exchanges = array();
  public static $queues = array();
  public static $consumers = array();
  
  public static function initialize_exchange($exchange_name, $flags = AMQP_PASSIVE, $type = AMQP_EX_TYPE_DIRECT, $arguments = array()) {
    
    // Verify that we have a valid connection
    if (!isset(self::$connection)) {
      self::$connection = new AMQPConnection(amqp_connection_args());
      self::$connection->connect();
      self::$channel = new AMQPChannel(self::$connection);
    }
    
    // Define Exchange (if necessary)
    if (!array_key_exists($exchange_name, self::$exchanges)) {
      $exchange = new AMQPExchange(self::$channel);
      // If the exchange name is '', then we want the default exchange,
      // which doesn't need to be initialized at all.
      if ($exchange_name != '') {
        try {
          $exchange->setName($exchange_name);
          $exchange->setArguments($arguments);
          $exchange->setFlags($flags);
          $exchange->setType($type);
          $exchange->declare();
        }
        catch (Exception $e) {
          watchdog('amqp', 'Unable to declare exchange: %error', array('%error' => $e->getMessage()));
        }
      }

      self::$exchanges[$exchange_name] = $exchange;
    }
  }

  public static function publish_all() {
    foreach (self::$messages as $exchange_name => $routing_key) {
      foreach ($routing_key as $key => $messages) {
        foreach ($messages as $message) {
          self::send_message($exchange_name, $key, $message['message'], $message['flags'], $message['type'], $message['arguments']);
        }
      }
    }
  }

  public static function send_message($exchange_name, $routing_key, $message, $flags = AMQP_PASSIVE, $type = AMQP_EX_TYPE_DIRECT, $arguments = array()) {
    self::initialize_exchange($exchange_name, $flags, $type, $arguments);
    self::$exchanges[$exchange_name]->publish($message, $routing_key);
  }

  public static function queue_message($exchange_name, $routing_key, $message, $flags = AMQP_PASSIVE, $type = AMQP_EX_TYPE_DIRECT, $arguments = array()) {
    $msg = array(
      'message' => $message,
      'arguments' => $arguments,
      'flags' => $flags,
      'type' => $type,
    );
    self::$messages[$exchange_name][$routing_key][] = $msg;
  }

  public static function register_consumers($consumers) {
    self::$consumers = $consumers;
  }

  public static function consume($exchange_name, $routing_key, $queue_name, $flags = AMQP_PASSIVE, $type = AMQP_EX_TYPE_DIRECT, $arguments = array()) {
    self::initialize_exchange($exchange_name, $flags, $type, $arguments);
    self::initialize_queue($exchange_name, $routing_key, $queue_name, $flags, $arguments);

    if ($message = self::$queues[$queue_name]->get(AMQP_NOPARAM)) {
      return $message;
    }
    return new AMQPEnvelope();
  }

  public static function initialize_queue($exchange_name, $routing_key, $queue_name, $flags = AMQP_PASSIVE, $arguments = array()) {
    if (!array_key_exists($queue_name, self::$queues)) {
      $queue = new AMQPQueue(self::$channel);
      $queue->setName($queue_name);
      $queue->setArguments($arguments);
      $queue->setFlags($flags);
      $queue->declare();
      $queue->bind($exchange_name, $routing_key);

      self::$queues[$queue_name] = $queue;
    }
  }

}
