/**
 * @file
 * Contains definition of the behaviour AnimateCSS AOS.
 */

(function ($, Drupal, drupalSettings, once) {
  "use strict";

  // Set AnimateCSS compat version for change classes.
  const compat = drupalSettings.animateCssAOS.compat;

  Drupal.behaviors.animateCssAosInit = {
    attach: function (context, settings) {

      // Set AnimateCSS and AOS records.
      const elements = drupalSettings.animateCssAOS.elements;
      let isItemInit = false;

      // Set AnimateCSS and AOS settings for each record.
      $.each(elements, function (index, element) {
        let options = {
          selector: element.selector,
          library: element.library,
          animation: element.animation,
          repeat: element.repeat,
          offset: element.offset,
          delay: element.delay,
          duration: element.duration,
          easing: element.easing,
          once: element.once,
          mirror: element.mirror,
          anchorPlacement: element.anchorPlacement,
        };

        // Prepares AOS if selector is available.
        if (once('aosjs', options.selector).length) {
          new Drupal.animateCssAosPrepare(options);
          // If there is even a selector then
          // the init value must be set to true.
          isItemInit = true;
        }
      });

      // Check is there item to init first.
      if ( isItemInit ) {
        // Set additional advanced setting for version 3.
        let additional = {
          initClassName: false,
          animatedClassName: (compat ? '' : 'animate__') + 'animated',
          useClassNames: true,
        };

        // Initial AOS now.
        AOS.init( additional );
      }

    }
  };

  Drupal.animateCssAosPrepare = function (options) {

    let prefix = compat ? '' : 'animate__'
      , animation = prefix + options.animation;

      // If the animation is of type 'In', sets the visibility to hidden
      // until reveal the element when scrolling and if the animation is
      // of type 'Out', sets the visibility to visible until the scroll
      // reaches the element, it will be removed from the display.
      if ((animation.indexOf("In") >= 0) || /In/.test(animation)) {
        $( options.selector ).css('visibility', 'hidden');
      } else {
        $( options.selector ).css('visibility', 'visible');
      }

      // Add Animate.css delay properties.
      if ( options.delay ) {
        $( options.selector ).css({
          '-webkit-animation-delay': options.delay + 'ms',
          '-moz-animation-delay': options.delay + 'ms',
          '-ms-animation-delay': options.delay + 'ms',
          '-o-animation-delay': options.delay + 'ms',
          'animation-delay': options.delay + 'ms',
          '--animate-delay': options.delay + 'ms',
        });
      }

      // Add Animate.css duration properties.
      if ( options.duration ) {
        $( options.selector ).css({
          '-webkit-animation-duration': options.duration + 'ms',
          '-moz-animation-duration': options.duration + 'ms',
          '-ms-animation-duration': options.duration + 'ms',
          '-o-animation-duration': options.duration + 'ms',
          'animation-duration': options.duration + 'ms',
          '--animate-duration': options.duration + 'ms',
        });
      }

      // Add Animate.css repeat class.
      if ( options.repeat && options.repeat != 'repeat-1' ) {
        $(options.selector).addClass(prefix + options.repeat);
      }

      // When the scroll reaches the desired element, if the animation
      // is type of "In", sets the visibility to visible.
      document.addEventListener('aos:in', ({ detail }) => {
        if ((animation.indexOf("In") >= 0) || /In/.test(animation)) {
          $( options.selector ).css('visibility', 'visible');
        }
      });

      // When the scroller passes the desired element, if the animation
      // is of type "In", it will set the visibility back to hidden so
      // that it is ready to repeat the animation if it is scrolled again.
      document.addEventListener('aos:out', ({ detail }) => {
        if ((animation.indexOf("In") >= 0) || /In/.test(animation)) {
          $( options.selector ).css('visibility', 'hidden');
        }
      });

    // Build AOS.js from animateCSS form settings.
    $( options.selector ).attr( "data-aos", animation );
    $( options.selector ).attr( "data-aos-offset", options.offset );
    $( options.selector ).attr( "data-aos-delay", options.delay );
    $( options.selector ).attr( "data-aos-duration", options.duration );
    $( options.selector ).attr( "data-aos-easing", options.easing );
    $( options.selector ).attr( "data-aos-anchor-placement", options.anchorPlacement );
    $( options.selector ).attr( "data-aos-once", options.once === 1);
    $( options.selector ).attr( "data-aos-mirror", options.mirror === 1);

  };

})(jQuery, Drupal, drupalSettings, once);
