<?php

namespace Drupal\anonymous_timezone;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\Core\Session\AccountProxy;
use GeoIp2\Database\Reader;
use GeoIp2\Exception\GeoIp2Exception;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Timezone logic for anonymous users.
 */
class AnonymousTimezoneAccountProxy extends AccountProxy {

  /**
   * The kill switch.
   *
   * @var \Drupal\Core\PageCache\ResponsePolicy\KillSwitch
   */
  protected KillSwitch $killSwitch;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * AccountProxy constructor.
   *
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   Event dispatcher.
   * @param \Drupal\Core\PageCache\ResponsePolicy\KillSwitch $killSwitch
   *   The kill switch.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(EventDispatcherInterface $eventDispatcher, KillSwitch $killSwitch, RequestStack $requestStack, ConfigFactoryInterface $configFactory, CacheBackendInterface $cache) {
    $this->eventDispatcher = $eventDispatcher;
    $this->killSwitch = $killSwitch;
    $this->requestStack = $requestStack;
    $this->configFactory = $configFactory;
    $this->cache = $cache;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccount() {
    $account = parent::getAccount();
    if ($account->isAnonymous()) {
      $tz = $this->getGeoTimeZone();
      if (!empty($tz)) {
        date_default_timezone_set($tz);
      }
    }
    return $account;
  }

  /**
   * {@inheritdoc}
   */
  public function getTimeZone() {
    if ($this->isAnonymous()) {
      $tz = $this->getGeoTimeZone();
      if (!empty($tz)) {
        return $tz;
      }
    }

    return parent::getTimeZone();
  }

  /**
   * Returns the timezone based on the IP address of the visitor.
   *
   * @return string|null
   *   Timezone ID or nothing.
   *
   * @throws \MaxMind\Db\Reader\InvalidDatabaseException
   */
  protected function getGeoTimeZone() {
    $settings = $this->configFactory->get('anonymous_timezone.settings');
    $current_request = $this->requestStack->getCurrentRequest();
    $exclude_paths = $settings->get('exclude_paths');
    if (!empty($exclude_paths) && is_array($exclude_paths)) {
      $uri = $current_request->getRequestUri();
      if (in_array($uri, $exclude_paths)) {
        return NULL;
      }
    }

    $this->killSwitch->trigger();

    $cached_data = $this->cache->get('anonymous_timezone');
    $ip = $current_request->getClientIp();
    $ips = [];
    if (!empty($cached_data) && array_key_exists($ip, $cached_data->data)) {
      return $cached_data->data[$ip];
    }
    if (isset($cached_data->data) && is_array($cached_data->data)) {
      $ips = $cached_data->data;
    }

    try {
      $geodb_file = $settings->get('geodb');
      if (empty($geodb_file) || !file_exists($geodb_file)) {
        throw new GeoIp2Exception('Missing GeoDB file');
      }
      $reader = new Reader($geodb_file);
      $record = $reader->city($ip);
      $tz = $record->location->timeZone;
      if (!empty($tz)) {
        $ips[$ip] = $tz;
      }
    }
    catch (GeoIp2Exception $e) {
      // No idea of the timezone at this point.
    }
    if (empty($ips[$ip])) {
      $ips[$ip] = NULL;
    }
    $this->cache->set('anonymous_timezone', $ips);
    return $ips[$ip];
  }

}
