<?php

/**
 * @file
 * Notification functions for the 'Answers' module
 * 
 * @author Chip Cleary
 * 
 */

 /* 
  * The file provides notification functions
  *    1. Notify question authors when their questions are answered (if they have so requested)
  *
  */

/*
 * Add settings to the notification form
 */ 
function _answers_notify_settings() {

  $form = array();
  
  $form['answers_notification'] = array(
    '#type' => 'fieldset',
    '#description' => t('Settings for the email which is sent to notify a question asker when a new answer is posted (if the asker asks for notification).'),
    '#title' => t('Notification Email Settings'),
  );

  $form['answers_notification']['answers_answer_notification_subject'] = array(
    '#type' => 'textfield',
    '#title' => t('Subject line'),
    '#default_value' => variable_get('answers_answer_notification_subject', ''),
    '#description' => t('Dynamic variables available: !question_user_name, !answer_user_name, !question_title, !question_url, and !site'),
    '#required' => TRUE
  );


  $form['answers_notification']['answers_answer_notification_body'] = array(
    '#type' => 'textarea',
    '#title' => t('Body'),
    '#default_value' => variable_get('answers_answer_notification_body', ''),
    '#description' => t('Dynamic variables available: !question_user_name, !answer_user_name, !question_title, !question_url, and !site.'),
    '#required' => TRUE
  );

  return $form;

}

/*
 * Pseudo implementation of hook_form_alter
 * (Note: This is not an actual hook but is instead called manually by answers_form_alter)
 *
 * If user is anonymous, hide the field allowing him/her to ask to be notified by email
 * This is a somewhat arduous way to do that based on http://drupal.org/node/208046
 */
function _answers_notify_form_alter(&$form, $form_state, $form_id) {
  global $user;
  
  // Hide the option to get notifications by email if the user is anonymous
  if ($form_id == 'question_node_form' && $user->uid == 0) { 
   $form['field_notify_p'][0]['#default_value']['value'] = 0;
   $form['field_notify_p']['#prefix'] = '<div style="display: none;">';
    $form['field_notify_p']['#suffix'] = '</div>';
  }
}

/*
 * Pseudo implementation of hook_nodeapi. 
 * (Note: This is not an actual hook but is instead called manually by answers_nodeapi)
 *
 * Tasks:
 *   - When a new answer is posted, notify the question author if appropriate.
 */
function _answers_notify_nodeapi(&$node, $op, $teaser) {
  switch ($op) {

    case 'insert':                        
      if ($node->type == 'answer') {
        $question_id = $node->field_answer_question[0]['nid']; // extract the nid of the question
        if ($question_id) {
          answers_notify($question_id);
        }
      }
      break;

  }
}


/*
 * If the question author should be notified, do so
 */
function answers_notify($nid) {
  
  global $user;
  
  $question = node_load($nid);
  $notify_p = $question->field_notify_p[0]['value']; // extract the nid of the question

  if ($notify_p) {
  
    $question_user = user_load($question->uid);

    $params = array(
      '!question_user_name'  => $question_user->name, 
      '!answer_user_name'    => $user->uid == 0 ? 'anonymous' : $user->name, 
      '!question_title'      => $question->title, 
      '!question_url'        => url('node/'. $nid, array('absolute' => TRUE, 'target' => '_blank')),
      '!site'                => variable_get('site_name', 'drupal'),
      );

    $subject_template = variable_get('answers_answer_notification_subject', '');
    $subject = t($subject_template, $params);

    $body_template = variable_get('answers_answer_notification_body', '');
    $body = t($body_template, $params);

    $headers['Mime-Version'] = '1.0';
    $headers['Content-Type'] = "text/html";
 
    $message = array(
      'headers' => $headers,
      'to'      => $question_user->mail,
      'from'    => variable_get('site_mail', ini_get('sendmail_from')),
      'subject' => $subject,
      'body'    => $body,
      );

    drupal_mail_send($message);

  };
}
