<?php

declare(strict_types=1);

namespace Drupal\ant_bulk\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\auto_node_translate\Form\TranslationForm;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\content_translation\ContentTranslationManager;
use Drupal\workflows\Entity\Workflow;
use Drupal\node\Entity\Node;

/**
 * Provides a Auto Node Translate Bulk form.
 */
final class TranslateForm extends TranslationForm {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The content translation manager.
   *
   * @var \Drupal\content_translation\ContentTranslationManager
   */
  protected $contentTranslationManager;

  /**
   * The content moderation information service.
   *
   * @var \Drupal\content_moderation\ModerationInformation|null
   */
  protected $contentModerationInformation;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    LanguageManagerInterface $language_manager,
    ConfigFactoryInterface $config,
    CurrentRouteMatch $route_match,
    TimeInterface $time,
    AccountProxyInterface $current_user,
    ModuleHandlerInterface $module_handler,
    EntityTypeManagerInterface $entity_type_manager,
    ContentTranslationManager $content_translation_manager,
    $content_moderation_information) {
    parent::__construct($language_manager, $config, $route_match, $time, $current_user, $module_handler);
    $this->entityTypeManager = $entity_type_manager;
    $this->contentTranslationManager = $content_translation_manager;
    $this->contentModerationInformation = $content_moderation_information;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('current_route_match'),
      $container->get('datetime.time'),
      $container->get('current_user'),
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('content_translation.manager'),
      $container->has('content_moderation.moderation_information') ? $container->get('content_moderation.moderation_information') : NULL
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ant_bulk_translate';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $node = NULL): array {
    $languages = $this->languageManager->getLanguages();
    $form['translate'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Languages to Translate'),
      '#closed' => FALSE,
      '#tree' => TRUE,
    ];
    $default_language_id = $this->languageManager->getDefaultLanguage()->getId();
    unset($languages[$default_language_id]);
    foreach ($languages as $language) {
      $languageId = $language->getId();
      $form['translate'][$languageId] = [
        '#type' => 'checkbox',
        '#title' => $this->t('@lang', [
          '@lang' => $language->getName(),
        ]),
      ];
    }
    $types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $allowed_types = array_filter($types, function ($type) {
      return $this->contentTranslationManager->isEnabled('node', $type);
    }, ARRAY_FILTER_USE_KEY);
    $form['bundles'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content types to translate'),
      '#tree' => TRUE,
    ];
    foreach ($allowed_types as $label => $entity) {
      $total = count($this->getDefaultNodes([$label]));
      $form['bundles'][$label] = [
        '#type' => 'checkbox',
        '#title' => $entity->label(),
        '#description' => $this->t("Total nodes: @total", [
          '@total' => $total,
        ]) . $this->getTotalTranslatedDescription($label),
      ];
    }
    if ($this->contentModerationInformation) {
      $form['workflow'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Workflows translation state'),
        '#tree' => TRUE,
      ];
      $workflows = Workflow::loadMultipleByType('content_moderation');
      foreach ($workflows as $label => $workflow) {
        $states = $workflow->getTypePlugin()->getStates();
        $form['workflow'][$label] = [
          '#type' => 'fieldset',
          '#title' => $workflow->label(),
        ];
        $form['workflow'][$label]['state'] = [
          '#type' => 'radios',
          '#options' => array_map(function ($state) {
            return $state->id();
          }, $states),
          '#title' => $this->t('Translation state'),
          '#default_value' => reset($states)->id(),
        ];
      }
    }
    $form['batch_size'] = [
      '#title' => $this->t('Batch size'),
      '#type' => 'number',
      '#min' => 0,
      '#description' => $this->t('Leave empty for all.'),
    ];
    $form['overwrite'] = [
      '#title' => $this->t('Overwrite translations'),
      '#type' => 'checkbox',
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('translate'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $bundles = $this->getCheckboxSelectedKeys($values['bundles']);
    $languages = $this->getCheckboxSelectedKeys($values['translate']);
    if (empty($bundles)) {
      $form_state->setErrorByName('bundles', $this->t('Please select at least one content type.'));
    }
    if (empty($languages)) {
      $form_state->setErrorByName('translate', $this->t('Please select at least one translation language.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $bundles = $this->getCheckboxSelectedKeys($values['bundles']);
    $languages = $values['translate'];
    $workflows = $values['workflow'] ?? NULL;
    $nodes = $this->getNodes($bundles, $values['batch_size'], $values['overwrite'], $languages);
    $operations[] = [
      [$this, 'translateBatch'],
      [count($nodes), array_values($nodes), $languages, $workflows],
    ];
    $batch = [
      'title' => $this->t('Translating ...'),
      'operations' => $operations,
      'finished' => [$this, 'translateFinished'],
    ];
    batch_set($batch);
  }

  /**
   * Translation batch.
   *
   * @param int $total
   *   The total number of items to be translated in the batch process.
   * @param array $nodes
   *   The nodes ids.
   * @param array $translations
   *   The languages ids to translate to.
   * @param mixed $workflows
   *   The languages ids to translate to.
   * @param mixed $context
   *   Information about the current batch process.
   */
  public function translateBatch($total, array $nodes, array $translations, $workflows, &$context) {
    if (empty($context['sandbox'])) {
      $context['sandbox']['progress'] = 0;
      $context['sandbox']['current_number'] = 0;
      $context['sandbox']['max'] = $total;
    }
    $batchSize = 1;
    $current = $context['sandbox']['progress'];
    for ($i = $current; $i < ($current + $batchSize); $i++) {
      if ($i < $total) {
        $node = $this->entityTypeManager->getStorage('node')->load($nodes[$i]);
        $title = $node->getTitle();
        $this->autoNodeTranslateNode($node, $translations);
        if ($this->contentModerationInformation && $this->contentModerationInformation->isModeratedEntity($node) && !empty($workflows)) {
          $this->setTranslationsModeratedState($node, $translations, $workflows);
        }
        $context['message'] = $this->t('translated @title with id:@id', [
          '@title' => $title,
          '@id' => $node->id(),
        ]);
        $context['results'][] = $node->id();
        $context['sandbox']['progress']++;
        $context['sandbox']['current_number'] = $i;
      }
    }
    if ($context['sandbox']['progress'] != $context['sandbox']['max']) {
      $context['finished'] = $context['sandbox']['progress'] / $context['sandbox']['max'];
    }
  }

  /**
   * Displays a message indicating the success or failure of the batch.
   *
   * @param bool $success
   *   Indicates whether the translate operation was successful or not.
   * @param mixed $results
   *   Array containing the results of the translate operation.
   * @param mixed $operations
   *   The array of operations that were executed during the translate process.
   */
  public function translateFinished($success, $results, $operations) {
    if ($success) {
      $message = $this->t('@count elements processed.', ['@count' => count($results)]);
    }
    else {
      $message = $this->t('Finished with an error.');
    }
    $this->messenger()->addStatus($message);
  }

  /**
   * Gets the keys of selecetd checkboxes.
   *
   * @param array $values
   *   An array of checkbox values in the form key => value.
   *   ex: ['article' => 1, 'blog' => 0].
   *
   * @return array
   *   An with the keys of the selected checkboxes.
   */
  private function getCheckboxSelectedKeys(array $values): array {
    $selected_values = array_filter($values, function ($value) {
      return $value == 1;
    });
    $selected_keys = array_keys($selected_values);
    return $selected_keys;
  }

  /**
   * Updates the moderation state of translations.
   *
   * @param \Drupal\node\Entity\Node $node
   *   The original node.
   * @param array $languages
   *   Information about the languages to translate to in the format:
   *   [language_id => to_translate] ex: ['fr => 0, 'pt-pt' => 1].
   * @param array $workflows
   *   Information about the default state for each workflow.
   */
  private function setTranslationsModeratedState(Node $node, array $languages, array $workflows) {
    $workflow = $this->contentModerationInformation->getWorkflowForEntity($node);
    $state_id = $workflows[$workflow->id()]['state'];
    $state = $workflow->getTypePlugin()->getStates()[$state_id];
    foreach ($languages as $languageId => $value) {
      if ($value) {
        $node_trans = $this->getTransledNode($node, $languageId);
        $node_trans->set('moderation_state', $state_id);
        $node_trans = $this->entityTypeManager->getStorage('node')->createRevision($node_trans, $state->isDefaultRevisionState());
        $state->isPublishedState() ? $node_trans->setPublished() : $node_trans->setUnpublished();
        $node_trans->save();
      }
    }
    $node->save();
  }

  /**
   * Retrieves nodes to be translated.
   *
   * @param array $bundles
   *   The content types of the nodes that you want to retrieve.
   * @param int $batch_size
   *   The number of nodes to retrive. If empty all nodes will be retrieved.
   * @param bool $overwrite
   *   Boolean that indicates if we want to overwrite current translations.
   * @param array $languages
   *   Array containing languages Ids to translate to.
   *
   * @return array
   *   A list of nodes based on the provided parameters.
   */
  public function getNodes(array $bundles, $batch_size, $overwrite, array $languages): array {
    $languages_ids = $this->getCheckboxSelectedKeys($languages);
    $nodes = $this->getDefaultNodes($bundles);
    if (!$overwrite) {
      $translated_nodes = $this->getTranslatedNodes($bundles, $languages_ids);
      $nodes = array_filter($nodes, function ($node) use ($translated_nodes) {
        return !in_array($node, $translated_nodes);
      });
    }
    if (!empty($batch_size)) {
      $nodes = array_slice($nodes, 0, intval($batch_size));
    }
    return $nodes;
  }

  /**
   * Retrieves all nodes of the specified types.
   *
   * @param array $bundles
   *   An array that contains the node types (bundle names) for which you want
   *   to retrieve the nodes.
   *
   * @return array
   *   An array of node IDs that match the specified node types.
   */
  private function getDefaultNodes(array $bundles): array {
    $nodes = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', $bundles, 'IN')
      ->sort('nid', 'DESC')
      ->execute();
    return $nodes;
  }

  /**
   * Retrieves all nodes of the specified types and languages.
   *
   * @param array $bundles
   *   An array that contains the node types (bundle names) for which you want
   *   to retrieve the nodes.
   * @param array $languages
   *   An array whith the language codes for which you want to retrieve
   *   translated nodes.
   *
   * @return array
   *   Returns an array of node IDs that meet the specified conditions.
   */
  private function getTranslatedNodes(array $bundles, array $languages): array {
    $nodes = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', $bundles, 'IN')
      ->condition('langcode', $languages, 'IN')
      ->sort('nid', 'DESC')
      ->execute();
    return $nodes;
  }

  /**
   * Returns the description for the translated nodes count.
   *
   * @param string $bundle
   *   The content type of the node.
   *
   * @return mixed
   *   The formated information about the total translations for each language.
   */
  private function getTotalTranslatedDescription($bundle) {
    $languages = $this->languageManager->getLanguages();
    $default_language_id = $this->languageManager->getDefaultLanguage()->getId();
    unset($languages[$default_language_id]);
    $description = "";
    foreach ($languages as $id => $language) {
      $total = $this->entityTypeManager->getStorage('node')->getQuery()
        ->accessCheck(FALSE)
        ->condition('type', $bundle)
        ->condition('langcode', $id)
        ->count()
        ->execute();
      $description .= $this->t("<br> Translated total for @language: @total", [
        '@language' => $language->getName(),
        '@total' => $total,
      ]);
    }
    return $description;
  }

}
