<?php

namespace Drupal\anti_duplicates\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\Node;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\AppendCommand;
use Drupal\Core\Ajax\RemoveCommand;
use Drupal\Core\Url;

/**
 * Hooks all node forms.
 */
class FormAlter {

  /**
   * Search for duplicates based on the title of the node being created.
   */
  public static function duplicatesSearch(array &$form, FormStateInterface $form_state) {

    $config = \Drupal::config('anti_duplicates.settings');

    // Get the node title.
    $title = $form_state->getValue('title')[0]['value'];
    $title = trim($title);

    $nodes = NULL;
    $count = 0;

    // Get the search type.
    switch ($config->get('anti_duplicates_search_type')) {
      case 0:
        $count = self::getNodesDefaultSearchType($title, $nodes);
        break;

      case 1:
        $title = '%' . $title . '%';
        $nids = \Drupal::entityQuery('node')->condition('title', $title, 'LIKE')->range(0, 5)->execute();
        $nodes = Node::loadMultiple($nids);
        $count = \Drupal::entityQuery('node')->condition('title', $title, 'LIKE')->count()->execute();
        break;

      case 2:
        $title_array = explode(' ', $title);
        $nids_query = \Drupal::entityQuery('node');
        $orGroup = $nids_query->orConditionGroup();
        foreach ($title_array as $key => $value) {
          $orGroup->condition('title', '%' . $value . '%', 'LIKE');
        }

        $nids = $nids_query->condition($orGroup)->range(0, 5)->execute();
        $nodes = Node::loadMultiple($nids);
        $count = \Drupal::entityQuery('node')->condition($orGroup)->count()->execute();
        break;

      default:
        $nodes = self::get_nodes_default_search_type($title);
        break;
    }

    // Add an Ajax response.
    $response = new AjaxResponse();

    // Append a list of found results.
    $response->addCommand(new RemoveCommand('#dw-listing'));
    $response->addCommand(new AppendCommand('#dw_container', '<ul id="dw-listing"></ul>'));

    $txt = t('Total');
    $response->addCommand(new RemoveCommand('#dw-total'));
    $response->addCommand(new AppendCommand('#dw_container', '<span id="dw-total">' . $txt . ' : <b>' . $count . '</b></span>'));

    // List all possible duplicates found.
    foreach ($nodes as $nid => $node) {
      $options = ['absolute' => TRUE];
      $url = Url::fromRoute('entity.node.canonical', ['node' => $nid], $options);
      $url = '<a href="' . $url->toString() . '" title="' . $node->title->value . '" target="_blank">' . $node->title->value . '</a>';
      $response->addCommand(new AppendCommand('#dw-listing', '<li>' . $url . '</li>'));
    }

    /*
    add a link to the anti-duplicates form when the option disable form is
    checked, this link allows the user to enable the form submission by
    agreeing that the content he is writing is unique.
     */
    if ($config->get('anti_duplicates_form_submission') && intval($count) > 0) {
      $response->addCommand(new RemoveCommand('#dw-enable-form'));
      $btn = '<a href="#" class="button" id="dw-enable-form">' . t('not a duplicate') . '</a>';
      $response->addCommand(new AppendCommand('#dw_container', $btn));
    }
    else {
      $response->addCommand(new RemoveCommand('#dw-enable-form'));
    }

    return $response;

  }

  /**
   * Get nodes for the default search mode.
   */
  private function getNodesDefaultSearchType($the_title, &$nodes) {
    $title = '%' . str_replace(' ', '%', $the_title) . '%';
    $query = \Drupal::entityQuery('node')->condition('title', $title, 'LIKE');
    $nids = $query->range(0, 5)->execute();
    $nodes = Node::loadMultiple($nids);

    return \Drupal::entityQuery('node')->condition('title', $title, 'LIKE')->count()->execute();
  }

}
