import { getPluralFormForCardinalByLocale } from 'fast-plural-rules';

/**
 * Converts an HTML hexadecimal color code into a CSS rgba format with optional opacity argument.
 * Credit: https://stackoverflow.com/a/51564734
 * @param  {string} hex       The color in hex format as defined in the MUI theme.
 * @param  {Number} [alpha=1] The value for the target alpha (opacity).
 * @return {string}           The rgba value ready to use as a CSS value.
 */
const hex2rgba = (hex, alpha = 1) => {
  const [r, g, b] = hex.match(/\w\w/g).map((x) => parseInt(x, 16));
  return `rgba(${r},${g},${b},${alpha})`;
};

/**
 * Debounces any function by a given timeout.
 * Credit: https://www.vhudyma-blog.eu/debounce-in-react/
 * @param  {Function} callback The function to call when the debouncer ends.
 * @param  {Number} [wait=300] How many milliseconds to wait until debouncing occurs.
 * @return {Function}          The debounced function.
 */
const debounce = (callback, wait = 300) => {
  let timeout;

  return (...args) => {
    clearTimeout(timeout);
    timeout = setTimeout(() => callback.apply(this, args), wait);
  };
};

const formatTime = (seconds) => {
  const date = new Date(seconds * 1000);
  const hh = date.getUTCHours();
  const mm = date.getUTCMinutes();
  const ss = pad(date.getUTCSeconds());
  if (hh) {
    return `${hh}:${pad(mm)}:${ss}`;
  }
  return `${mm}:${ss}`;
};

const pad = (string) => {
  return ('0' + string).slice(-2);
};

// Parse entity_labels only once and store here.
let entityLabels = null;

const getEntityLabel = (entity, count = 1) => {
  if (!entityLabels) {
    // Get courses page element and its content generated by Drupal.
    const element = document.getElementById('anu-application');
    entityLabels = JSON.parse(element.dataset.entity_labels);
  }

  const localizeMessage = (entity, count) => {
    const locale = drupalSettings.path.currentLanguage;
    const pluralForm = getPluralFormForCardinalByLocale(locale, count);
    return entity.split('|')[pluralForm];
  };

  if (entity === 'Course page') {
    return localizeMessage(entityLabels.courses_page_labels, count);
  } else if (entity === 'Course') {
    return localizeMessage(entityLabels.course_labels, count);
  } else if (entity === 'Lesson') {
    return localizeMessage(entityLabels.lesson_labels, count);
  } else if (entity === 'Quiz') {
    return localizeMessage(entityLabels.assessment_labels, count);
  } else if (entity === 'Module') {
    return localizeMessage(entityLabels.module_labels, count);
  }

  return entity;
};

export { debounce, formatTime, hex2rgba, getEntityLabel };
