<?php

namespace Drupal\api_browser\Entity;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\SortArray;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Transliteration\PhpTransliteration;
use Drupal\Core\Url;
use Drupal\project_browser\ActivationStatus;
use Drupal\project_browser\ProjectBrowser\Project;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use function JmesPath\search;

/**
 * Defines api_browser service entity type.
 *
 * @ConfigEntityType(
 *   id = "api_browser_service",
 *   label = @Translation("API Browser Service"),
 *   label_collection = @Translation("API Browser Services"),
 *   label_singular = @Translation("API Browser Service"),
 *   label_plural = @Translation("API Browser Services"),
 *   label_count = @PluralTranslation(
 *     singular = "@count API Browser Service",
 *     plural = "@count API Browser Services",
 *   ),
 *   handlers = {
 *     "access" = "Drupal\api_browser\ApiBrowserServiceAccessControlHandler",
 *     "list_builder" = "Drupal\api_browser\ApiBrowserServiceListBuilder",
 *     "form" = {
 *       "add" = "Drupal\api_browser\Form\ApiBrowserServiceForm",
 *       "edit" = "Drupal\api_browser\Form\ApiBrowserServiceForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     }
 *   },
 *   config_prefix = "service",
 *   admin_permissions = "administer api_browser_service",
 *   links = {
 *     "collection" = "/admin/config/development/project_browser/api_browser/service",
 *     "add-form" = "/admin/config/development/project_browser/api_browser/service/add",
 *     "edit-form" = "/admin/config/development/project_browser/api_browser/service/{api_browser_service}",
 *     "delete-form" = "/admin/config/development/project_browser/api_browser/service/{api_browser_service}/delete"
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "id",
 *     "name",
 *     "description",
 *     "categories",
 *     "listing",
 *     "search",
 *     "project",
 *     "field_mapping"
 *   }
 * )
 */
class ApiBrowserService extends ConfigEntityBase {

  use StringTranslationTrait;

  /**
   * Guzzle Client used for making requests.
   *
   * @var \GuzzleHttp\Client|null
   */
  protected ?Client $httpClient = NULL;

  /**
   * Logger used for logging exceptions.
   *
   * @var \Psr\Log\LoggerInterface|null
   */
  protected ?LoggerInterface $logger = NULL;

  /**
   * Cache Backend Service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|null
   */
  protected ?CacheBackendInterface $cacheBackend = NULL;

  /**
   * Transliteration Service.
   *
   * @var \Drupal\Core\Transliteration\PhpTransliteration|null
   */
  protected ?PhpTransliteration $transliteration = NULL;

  /**
   * Renderer Service.
   *
   * @var \Drupal\Core\Render\RendererInterface|null
   */
  protected ?RendererInterface $renderer = NULL;

  /**
   * Return the logging interface.
   *
   * @return \Psr\Log\LoggerInterface
   *   Logging Channel.
   */
  protected function getLogger(): LoggerInterface {
    if (!$this->logger) {
      $this->logger = \Drupal::logger('api_browser');
    }
    return $this->logger;
  }

  /**
   * Return a list of categories.
   *
   * @return array
   *   Return a list of categories.
   */
  public function getCategories(): array {
    $data = $this->splitToArray($this->get('categories') ?? '');
    $list = [];
    foreach ($data as $id => $label) {
      $list[] = ['id' => $id, 'name' => $label];
    }
    uasort($list, fn($a, $b) => SortArray::sortByKeyString($a, $b, 'name'));
    return $list;
  }

  /**
   * Split the string into a keyed array.
   *
   * @param string $data
   *   Data to parse.
   * @param string $delimiter
   *   Delimiter to use for parsing the data.
   *
   * @return array
   *   Return data with information.
   */
  protected function splitToArray(string $data, string $delimiter = '|'): array {
    $result = [];
    $data = str_ireplace(["\r\n", "\r", "\n"], PHP_EOL, $data);
    $lines = explode(PHP_EOL, $data);
    foreach ($lines as $line) {
      $parts = str_getcsv($line, $delimiter);
      if (count($parts) >= 2) {
        $key = trim($parts[0]);
        $value = trim($parts[1]);
        $result[$key] = $value;
      }
      elseif (count($parts) === 1) {
        $key = trim($parts[0]);
        $result[$key] = $key;
      }
    }

    return $result;
  }

  /**
   * Returning the listing endpoint.
   */
  public function getListingEndpoint(): string {
    return $this->get('listing')['endpoint'] ?? '';
  }

  /**
   * Returning the listing request method.
   */
  public function getListingMethod(): string {
    return $this->get('listing')['method'] ?? '';
  }

  /**
   * Returning the listing request auth type.
   */
  public function getListingAuthType(): string {
    return $this->get('listing')['auth'] ?? '';
  }

  /**
   * Return the listing credentials.
   */
  public function getListingAuthCredentials(): array {
    return $this->get('listing')['credentials'] ?? [];
  }

  /**
   * Return the listing path.
   */
  public function getListingResultsPath(): string {
    return $this->get('listing')['path'];
  }

  /**
   * Return if pagination is enabled for listing endpoint.
   */
  public function getListingPaginationEnabled(): bool {
    return $this->get('listing')['pagination'] ?? FALSE;
  }

  /**
   * Return the listing page variable.
   */
  public function getListingPageVariable(): string {
    return $this->get('listing')['page'];
  }

  /**
   * Return the listing per_page variable.
   */
  public function getListingPerPageVariable(): string {
    return $this->get('listing')['per_page'] ?? '';
  }

  /**
   * Return the Query parameters.
   */
  public function getSearchParams(): array {
    return $this->get('search') ?? [];
  }

  /**
   * Return if the project endpoint is enabled.
   */
  public function getProjectEndpointEnabled(): bool {
    return $this->get('project')['enable'] ?? FALSE;
  }

  /**
   * Return the project endpoint.
   */
  public function getProjectEndpoint(): string {
    return $this->get('project')['endpoint'] ?? '';
  }

  /**
   * Return the project method.
   */
  public function getProjectMethod(): string {
    return $this->get('project')['method'] ?? '';
  }

  /**
   * Return the project auth type.
   */
  public function getProjectAuthType(): string {
    return $this->get('project')['auth'] ?? '';
  }

  /**
   * Return the project auth credentials.
   */
  public function getProjectAuthCredentials(): array {
    return $this->get('project')['credentials'] ?? [];
  }

  /**
   * Return the project auth results path.
   */
  public function getProjectResultsPath(): string {
    return $this->get('project')['path'] ?? '';
  }

  /**
   * Return the project field mapping.
   */
  public function getFieldMapping(): array {
    return $this->get('field_mapping') ?? [];
  }

  /**
   * Return the field mapping field value.
   */
  public function getFieldMappingField(string $field): ?string {
    return $this->get('field_mapping')[$field] ?? NULL;
  }

  /**
   * Get the Guzzle Client to act on.
   *
   * @return \GuzzleHttp\Client
   *   Return the created Client
   */
  protected function getClient(): Client {
    if (!$this->httpClient) {
      $this->httpClient = \Drupal::httpClient();
    }
    return $this->httpClient;
  }

  /**
   * Return a list of all projects from service.
   *
   * @param array $query
   *   Query parameters to use for searching.
   *
   * @return \Drupal\project_browser\ProjectBrowser\Project[]
   *   Return an array of Projects.
   */
  public function getList(array $query = []): array {
    return $this->getProjectList($query);
  }

  /**
   * Query the end point and return the list of projects.
   *
   * Example of the data passed in the query.
   *
   * array (
   *   'page' => 0,
   *   'limit' => 12,
   *   'sort' => 'a_z',
   *   'maintenance_status' => 'all',
   *   'development_status' => 'all',
   *   'security_advisory_coverage' => 'covered',
   *   'source' => 'api_browser_project:example_packagist',
   *   'categories' => '',
   * )
   *
   * @param array $query
   *   Array of data passed in by Project Browser query.
   *
   * @return \Drupal\project_browser\ProjectBrowser\Project[]
   *   Return the array of the projects.
   */
  protected function getProjectList(array $query = []): array {
    $cid = $this->id() . ':results';
    $cache = $this->getCacheBackend()->get($cid);
    if ($cache) {
      $results = $cache->data;
    }
    else {
      try {
        $results = $this->queryListingEndpoint($query);
      }
      catch (\Throwable $exception) {
        $this->getLogger()->error($exception->getMessage());
        $results = [];
      }

      if (!empty($results)) {
        $this->getCacheBackend()->set(
          $cid,
          $results,
          Cache::PERMANENT,
          [
            'api_browser:results',
            'api_browser:results:' . $this->id(),
          ]
        );
      }
    }

    $list = [];
    foreach ($results as $key => $item) {
      if (!array_key_exists('item_key', $item)) {
        $item['item_key'] = $key;
      }
      if ($project = $this->getProject($item)) {
        $list[] = $project;
      }
    }
    return $list;
  }

  /**
   * Query the listing endpoint to return results.
   *
   * @param array $query
   *   Array of data passed in by Project Browser query.
   *
   * @return array
   *   Data returned from the listing endpoint.
   */
  protected function queryListingEndpoint(array $query = []): array {
    $request = $this->makeListingRequest($query);
    $response = $this->getClient()->sendRequest($request);
    $items = Json::decode($response->getBody()->getContents());
    $this->moduleHandler()->alter(
      ['api_project_project_list_' . $this->id(), 'api_browser_project_list'],
      $items
    );

    $results = search($this->getListingResultsPath(), $items) ?? [];
    if (count($results) > $query['limit'] && $this->getListingPaginationEnabled()) {
      $query['page'] = $query['page'] + 1;
      $results = array_merge($results, $this->queryListingEndpoint($query));
    }

    return $results;
  }

  /**
   * Make the listing request and return the request.
   *
   * @param array $query
   *   Query of data being requested.
   *
   * @return \Psr\Http\Message\RequestInterface
   *   Return the request.
   */
  protected function makeListingRequest(array $query = []): RequestInterface {
    $variables = [];
    $options = [];

    $credentials = $this->getListingAuthCredentials();
    $authType = $this->getListingAuthType();
    $options = $this->authorizeRequest($authType, $credentials, $options);

    if ($this->getListingPaginationEnabled()) {
      if (
        !empty($pageVariable = $this->getListingPageVariable()) &&
        isset($query['page'])
      ) {
        $variables[$pageVariable] = $query['page'];
      }

      if (
        !empty($perPageVariable = $this->getListingPerPageVariable()) &&
        isset($query['limit'])
      ) {
        $variables[$perPageVariable] = $query['limit'];
      }
    }

    foreach ($this->get('search') as $searchParam) {
      $variables[$searchParam['parameter']] = $searchParam['value'];
    }

    // Depending on the type of method will be how we send the data.
    switch ($this->getListingMethod()) {
      case 'POST':
        $options['body'] = $variables;
        break;

      case 'GET':
      default:
        $options['query'] = array_merge($options['query'] ?? [], $variables);
        break;
    }

    $endpoint = $this->getListingEndpoint();
    $this->moduleHandler()->alter(
      ['api_browser_project_list_request_' . $this->id(), 'api_browser_project_list_request'],
      $options,
      $endpoint
    );

    $endpoint = Url::fromUri($endpoint, $options)->toString();
    return new Request(
      $this->getListingMethod(),
      $endpoint,
        $options['headers'] ?? [],
        $options['body'] ?? NULL
    );
  }

  /**
   * Return the response from the project request.
   *
   * @param array $record
   *   Record of data to check.
   *
   * @return \Psr\Http\Message\RequestInterface
   *   Return the response.
   */
  protected function makeProjectRequest(array $record = []): RequestInterface {
    $options = [];
    $credentials = $this->getProjectAuthCredentials();
    $authType = $this->getProjectAuthType();
    $options = $this->authorizeRequest($authType, $credentials, $options);

    $endpoint = $this->renderTwigTemplate($this->getProjectEndpoint(), $record);

    $this->moduleHandler()->alter(
      ['api_browser_project_request_' . $this->id(), 'api_browser_project_request'],
      $options,
      $endpoint,
      $variables
    );
    $endpoint = Url::fromUri($endpoint, $options)->toString();
    return new Request(
      $this->getProjectMethod(),
      $endpoint,
      $options['headers'] ?? [],
      $options['body'] ?? NULL
    );
  }

  /**
   * Generate the project results array.
   *
   * @param string|array $record
   *   Record of data to use.
   *
   * @return array
   *   Array of data.
   */
  protected function getProjectResult(string|array $record = []): array {
    $result = is_array($record) ? $record : ['item_key' => $record];

    if ($this->getProjectEndpointEnabled()) {
      $request = $this->makeProjectRequest($result);
      $response = $this->getClient()->sendRequest($request);
      $item = Json::decode($response->getBody()->getContents());
      $this->moduleHandler()->alter(
        ['api_browser_project_info_' . $this->id(), 'api_browser_project_info'],
        $item
      );

      $path = $this->renderTwigTemplate($this->getProjectResultsPath(), $record);
      $results = search($path, $item);
      $result = array_merge($result, $results);
    }

    return $result;
  }

  /**
   * Query the project endpoint.
   *
   * @param string|array $record
   *   Information related to the project.
   *
   * @return array
   *   Return structured data for the project.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   *   Throws GuzzleException if errors.
   */
  protected function queryProjectEndpoint(string|array $record = []): array {
    $result = $this->getProjectResult($record);

    $projectInfo = [
      'logo' => $this->fieldValue('logo', $result),
      'isCompatible' => boolval($this->fieldValue('compatible', $result)),
      'isMaintained' => boolval($this->fieldValue('maintained', $result)),
      'isCovered' => boolval($this->fieldValue('covered', $result)),
      'isActive' => boolval($this->fieldValue('active', $result)),
      'starUserCount' => intval($this->fieldValue('star_user_count', $result)),
      'projectUsageTotal' => intval($this->fieldValue('project_usage_total', $result)),
      'machineName' => $this->getMachineName($this->fieldValue('machine_name', $result)),
      'body' => [
        'summary' => $this->fieldValue('short_description', $result),
        'value' => $this->fieldValue('long_description', $result),
      ],
      'title' => $this->fieldValue('title', $result),
      'changed' => intval($this->fieldValue('changed', $result)),
      'created' => intval($this->fieldValue('created', $result)),
      'author' => $this->explodeString($this->fieldValue('author', $result)),
      'packageName' => $this->fieldValue('package_name', $result),
      'url' => $this->fieldValue('url', $result),
      'categories' => $this->explodeString($this->fieldValue('categories', $result)),
      'images' => $this->explodeString($this->fieldValue('images', $result)),
      'warnings' => $this->explodeString($this->fieldValue('warnings', $result)),
      'type' => $this->fieldValue('type', $result),
    ];
    if (!empty($projectInfo['logo']) && filter_var($projectInfo['logo'], FILTER_VALIDATE_URL)) {
      $projectInfo['logo'] = [
        'file' => [
          'uri' => $projectInfo['logo'],
          'resource' => 'image',
        ],
        'alt' => $projectInfo['title'],
      ];
    }
    else {
      $projectInfo['logo'] = [];
    }
    return $projectInfo;
  }

  /**
   * Get the field value based on the twig value provided.
   *
   * @param string $fieldName
   *   Field name to get.
   * @param array $result
   *   Results to use for replacement.
   *
   * @return string
   *   Return the string.
   */
  protected function fieldValue(string $fieldName, array $result = []): string {
    return $this->renderTwigTemplate($this->getFieldMappingField($fieldName), $result);
  }

  /**
   * Get the Project object.
   *
   * @param array $record
   *   Details from the list process.
   *
   * @return \Drupal\project_browser\ProjectBrowser\Project|null
   *   Return the generated project.
   */
  protected function getProject(array $record = []): ?Project {
    $project_hash = md5(Json::encode($record));
    $cid = $this->id() . ':result:' . $project_hash;
    $cache = $this->getCacheBackend()->get($cid);
    if ($cache) {
      $projectInfo = $cache->data;
    }
    else {
      try {
        $projectInfo = $this->queryProjectEndpoint($record);
      }
      catch (\Throwable $exception) {
        $this->getLogger()->error($exception->getMessage());
        return NULL;
      }

      $this->getCacheBackend()->set(
        $cid,
        $projectInfo,
        Cache::PERMANENT,
        [
          'api_browser:results',
          'api_browser:results:' . $this->id(),
          'api_browser:project_info',
          'api_browser:project_info:' . $this->id(),
          'api_browser:project_info:' . $this->id() . ':' . $project_hash,
        ]
      );
    }

    $project = new Project(
      ...(array_values($projectInfo))
    );

    $this->moduleHandler()->alter(
      ['api_browser_project_browser_project_' . $this->id(), 'api_browser_project_browser_project'],
      $project,
      $result
    );

    return $project;
  }

  /**
   * Explode the string into an array based on the provide delimiter.
   *
   * @param string $data
   *   String to analyze.
   * @param string $delimiter
   *   String to use for delimiter.
   *
   * @return array
   *   Return array of strings.
   */
  protected function explodeString(string $data, string $delimiter = ','): array {
    return array_filter(explode($delimiter, $data));
  }

  /**
   * Set the appropriate details based on the auth type.
   *
   * @param string $authType
   *   Authorization type being requested.
   * @param array $credentials
   *   Array of credentials stored within the config.
   * @param array $options
   *   Options to use for the guzzle request.
   *
   * @return array
   *   Return the authorization header.
   */
  protected function authorizeRequest(string $authType, array $credentials = [], array $options = []): array {
    switch ($authType) {
      case 'api_key':
        switch ($credentials['location']) {
          case 'header':
          case 'query':
            $options[$credentials['location']][$credentials['key']] = $credentials['value'];
            break;
        }
        break;

      case 'bearer_token':
        $options['header']['Authorization'] = 'Bearer ' . $credentials['token'];
        break;

      case 'basic_auth':
        $options['auth'] = [
          $credentials['username'] ?? '',
          $credentials['password'] ?? '',
        ];
        break;
    }

    $this->moduleHandler()->alter(
      ['api_browser_authorize_request_' . $this->id(), 'api_browser_authorize_request'],
      $options,
      $authType,
      $credentials
    );

    return $options;
  }

  /**
   * Render the twig template based on the variables.
   *
   * @param string $template
   *   Template to compute.
   * @param array $context
   *   Context or variables to use for replacement.
   *
   * @return string
   *   Return the rendered string.
   */
  protected function renderTwigTemplate(string $template, array $context = []): string {
    try {
      $build = [
        '#type' => 'inline_template',
        '#template' => '{% apply spaceless %}' . $template . '{% endapply %}',
        '#context' => $context,
      ];
      // Render twig template.
      $output = $this->getRenderer()->renderInIsolation($build);
      // Remove extra spaces to make a single line.
      $output = preg_replace('#(\s){2,}#', ' ', $output);
    }
    catch (\Exception $exception) {
      $this->getLogger()->error($exception->getMessage());
      $output = '';
    }
    return trim($output);
  }

  /**
   * Generates a machine name from a string.
   *
   * This is basically the same as what is done in
   * \Drupal\Core\Block\BlockBase::getMachineNameSuggestion() and
   * \Drupal\system\MachineNameController::transliterate(), but it seems
   * that so far there is no common service for handling this.
   *
   * @param string $string
   *   String to turn into a machine name.
   *
   * @return string
   *   Return the machine name version of the string.
   *
   * @see \Drupal\Core\Block\BlockBase::getMachineNameSuggestion()
   * @see \Drupal\system\MachineNameController::transliterate()
   */
  protected function getMachineName(string $string): string {
    $transliterated = $this->getTransliteration()->transliterate($string, LanguageInterface::LANGCODE_DEFAULT, '_');
    $transliterated = mb_strtolower($transliterated);
    $transliterated = preg_replace('@[^a-z0-9_.]+@', '_', $transliterated);
    return $transliterated;
  }

  /**
   * Return the cache service.
   *
   * @return \Drupal\Core\Cache\CacheBackendInterface
   *   Cache backend service.
   */
  protected function getCacheBackend(): CacheBackendInterface {
    if (!$this->cacheBackend) {
      $this->cacheBackend = \Drupal::cache();
    }
    return $this->cacheBackend;
  }

  /**
   * Return renderer service.
   *
   * @return \Drupal\Core\Render\RendererInterface
   *   Renderer service.
   */
  protected function getRenderer(): RendererInterface {
    if (!$this->renderer) {
      $this->renderer = \Drupal::service('renderer');
    }
    return $this->renderer;
  }

  /**
   * Return transliteration service.
   *
   * @return \Drupal\Core\Transliteration\PhpTransliteration
   *   Transliteration service.
   */
  protected function getTransliteration(): PhpTransliteration {
    if (!$this->transliteration) {
      $this->transliteration = \Drupal::transliteration();
    }
    return $this->transliteration;
  }

  /**
   * Query the endpoint and return the output.
   *
   * @return string
   *   Return the output from the query.
   */
  public function testListingEndpoint(): string {
    $output = [];

    $output[] = $this->t('Request Date/Time: @date', ['@date' => date('c')]);

    $defaultQuery = ['page' => 0, 'limit' => 12];
    try {
      $request = $this->makeListingRequest($defaultQuery);
      $output[] = $this->t('Listing URL: @url', ['@url' => $request->getUri()->__toString()]);

      $jsonResponse = Json::decode($this->getClient()->sendRequest($request)->getBody()->getContents());
      $json = json_encode($jsonResponse, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
      $output[] = $this->t('Listing HTTP Response:\r\n@response', ['@response' => $json]);

      $response = $this->queryListingEndpoint($defaultQuery);
      $json = json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
      $output[] = $this->t('Project Listing Result:\r\n@result', ['@result' => $json]);
    }
    catch (\Throwable $exception) {
      $output[] = $this->t('Error: @message', ['@message' => $exception->getMessage()]);
    }

    return implode(PHP_EOL . PHP_EOL, $output);
  }

  /**
   * Query the endpoint and return the output.
   *
   * @return string
   *   Return the output from the query.
   */
  public function testProjectEndpoint(): string {
    $output = [];

    $output[] = $this->t('Request Date/Time: @date', ['@date' => date('c')]);

    $defaultQuery = ['page' => '0', 'limit' => '12'];
    try {
      $projects = $this->queryListingEndpoint($defaultQuery);
      $firstKey = array_key_first($projects);

      $project = is_array($projects[$firstKey]) ? $projects[$firstKey] : ['item_key' => $projects[$firstKey]];
      $project = array_key_exists('item_key', $project) ? $project : array_merge(['item_key' => $firstKey], $project);

      if ($this->getProjectEndpointEnabled()) {
        $request = $this->makeProjectRequest($project);
        $output[] = $this->t('Sample Project URL: @url', ['@url' => $request->getUri()->__toString()]);

        $response = $this->getClient()->sendRequest($request);
        $json = json_encode(Json::decode($response->getBody()->getContents()), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $output[] = $this->t('Project HTTP Response:\r\n@result', ['@result' => $json]);
      }

      $projectResult = $this->getProjectResult($project);
      $json = json_encode($projectResult, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
      $output[] = $this->t('Project Results:\r\n@result', ['@result' => $json]);

      $response = $this->queryProjectEndpoint($project);
      $json = json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
      $output[] = $this->t('Project Field Mapping:\r\n@result', ['@result' => $json]);

      $projectObject = new Project(...array_values($response));
      $projectObject->commands = '';
      $projectObject->status = ActivationStatus::Absent;
      $projectObject->source = 'api_browser';
      $json = json_encode($projectObject, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
      $output[] = $this->t('Project Browser Project:\r\n@result', ['@result' => $json]);
    }
    catch (\Throwable $exception) {
      $output[] = $this->t('Error: @message', ['@message' => $exception->getMessage()]);
    }

    return implode(PHP_EOL . PHP_EOL, $output);
  }

}
